#!/bin/bash

. $(dirname $0)/init-functions

CHECK_BRANCH_NAME="kylinos-next stable-52-sp3 stable-25 stable-23"

function check_branch()
{
	for branch in $CHECK_BRANCH_NAME
	do
		git branch | grep $branch > /dev/null
		if [ $? = 1 ]; then
			echo -e "Local branch $branch not found, please set."
			return -1;
		fi
	done

	return 0;
}

function usage()
{
	echo -e "`basename $0` [OPTION] {CVE-2022-0001}"
	echo
	echo -e "  -h, --help\t\t\tshow this help info"
	echo -e "  -f, --file\t\t\tCVE file"
	echo -e "  -p, --pull\t\t\tpull project"
	echo -e "  -o, --output\t\toutput results"
	echo
	echo "Example:"
	echo "  cve-check.sh CVE-2022-0001 CVE-2022-0002 CVE-2022-0003"
}

function pull_branch()
{
	remote_branchs=$(git branch -a | grep remote | grep -v "cloud\|HEAD\|gerrit")
	for i in $CHECK_BRANCH_NAME
	do
		_branch=$(echo "$remote_branchs" | grep $i | head -n1)
		git branch $i --track ${_branch#remotes/origin/} 2>/dev/null;
	done

	git pull --rebase --autostash --tags
}

function write_file()
{
	[ ! -z "$OUTPUT_FILE" ] && echo -e "$1: [$2]" >> $OUTPUT_FILE

	echo -e "$2"
}

function commit-in-tree()
{
	fullhash=$(git rev-parse $1)
	if [ "$(git merge-base $fullhash $2)" = "$fullhash" ]; then
		echo "$fullhash"
		return 1
	fi

	# Grab the subject, since commit sha1 is different between branches we
	# have to look it up based on subject.
	subj=$(git log -1 --pretty="%s" $1)
	if [ $? -gt 0 ]; then
		return 0
	fi

	# Try and find if there's a commit with given subject the hard way
	for i in $(git log --pretty="%H" -F --grep "$subj" $STABLE_BRANCH_FULLHASH..$2); do
		cursubj=$(git log -1 --format="%s" $i)
		if [ "$cursubj" = "$subj" ]; then
			echo "$i"
			return 1
		fi
	done

	return 0
}

[ "$#" -lt 1 ] && usage && exit -1;

ARGS=`getopt -o hpf:o: -a --long help,file:,pull,output: -- "$@"`
[ $? != 0 ] && echo "Terminating..." && usage && exit -1

eval set -- "${ARGS}"

while true
do
	case $1 in
		-h | --help)
			usage
			exit 0;;
		-f | --file)
			FILENAME=$2
			shift 2;;
		-p | --pull)
			NEED_PULL=true
			shift 1;;
		-o | --output)
			OUTPUT_FILE=$2
			echo -n > $OUTPUT_FILE
			shift 2;;
		--)
			shift
			break;;
	esac
done

function handle_local_fix()
{
	LOCAL_FIX=$(git log --oneline --grep=$1 $STABLE_BRANCH_FULLHASH.. | tail -n1)
	if ! [ -z "${LOCAL_FIX}" ]; then
		C_HASH=$(echo $LOCAL_FIX | awk -F' ' '{print $1}')
		C_TITLE=$(git log --pretty=%s -1 $C_HASH)
		write_file $1 "Found fixed ${GREEN}commit $C_HASH (\"$C_TITLE\")${NC}, please confirm manually."
	fi
}

function do_check()
{
	echo "==========================================================="
	echo -e "${YELLOW}$1:${NC}"

	CVESITE="https://ubuntu.com/security/$1"

	BRK_FIX=`curl $CVESITE 2>&1 |grep "Fixed by\|Introduced by" -A 1 | sed -n 's/.*org\/linus\/\([0-z]*\).*/\1/p'`
	if [ -z "$BRK_FIX" ]; then
		write_file $1 "${YELLOW}Not found patch from $CVESITE, Please do it manually.${NC}" >&2
		return -1
	fi

	BRK_ID=$(echo $BRK_FIX | awk '{print $1}')
	FIX_ID=$(echo $BRK_FIX | awk '{print $2}')
	if [ -z "$FIX_ID" ]; then
		write_file $1 "Not found fixed commit id."
		return -1
	fi

	FIX_TITLE=$(git log --pretty=%s -1 $FIX_ID 2>/dev/null)
	if [ $? != 0 ]; then
		write_file $1 "Commit ID($FIX_ID) which fixed $CVE isn't a upstream ID, maybe there isn't upstream fixed patch!" >&2
		return -1
	fi

	BRK_TITLE=$(git log --pretty=%s -1 $BRK_ID 2>/dev/null)
	if [ $? -ne 0 ]; then
		write_file $1 "Commit ID($BRK_ID) which introduced $CVE isn't a upstream ID"
		return -1
	fi

	SHORT_FIX_ID=$(git log --pretty=%h -1 $FIX_ID)
	SHORT_BRK_ID=$(git log --pretty=%h -1 $BRK_ID)
	FIX_TAG=$(gdct $SHORT_FIX_ID)
	BRK_TAG=$(gdct $SHORT_BRK_ID)

	printf "Introduced by:\t%s %-23s %s\n" "${BRK_ID:0:13}" "(${BRK_TAG})" "${BRK_TITLE}"
	printf "Fixed by:\t%s %-23s %s\n" "${FIX_ID:0:13}" "(${FIX_TAG})" "${FIX_TITLE}"

	# Introduced patch is before.
	TAG_ID=$(git log $FIX_TAG --oneline -1 | awk -F' ' '{print $1}')
	fullhash=$(git rev-parse $TAG_ID)
	if [ "$(git merge-base $fullhash $STABLE_BRANCH_FULLHASH)" = "$fullhash" ]; then
		write_file $1 "Fixed patch tag is ${FIX_TAG} which is before V10 SPx, ${RED} $CVE didn't affect ${NC}"
		return 0
	fi

	# Init temporary variable
	fixed=true
	affect=false

	for BRANCH in ${CHECK_BRANCH_NAME};
	do
		B_HEAD=$(commit-in-tree $SHORT_FIX_ID origin/$BRANCH)
		if [ -z "$B_HEAD" ]; then
			# Check if we have the introdeced patch
			fullhash=$(git rev-parse $SHORT_BRK_ID)
			if [ "$(git merge-base $fullhash ${STABLE_BRANCH_FULLHASH}) = ${STABLE_BRANCH_FULLHASH}" ]; then
				commit-in-tree $SHORT_BRK_ID "origin/$BRANCH" > /dev/null
				if [ $? -eq 1 ]; then
					if [ $SHORT_BRK_ID = "1da177e4c3f4" ]; then
						write_file $1 "$(branch2name ${BRANCH}):\t$1 ${RED}not fixed, but don't find break commit id, handle it yourself.${NC}" >&2
					else
						write_file $1 "$(branch2name ${BRANCH}):\t$1 ${RED}not fixed, have break commit $SHORT_BRK_ID ${NC}" >&2
					fi
					fixed=false
				else
					echo -e "$(branch2name ${BRANCH}):\t${GREEN}Not affected, we don't have broken patch ${BRK_HEAD}(${BRK_TAG})${NC}"
				fi
			else
				write_file $1 "$(branch2name ${BRANCH}):\t${YELLOW} not fixed ${NC}"
				fixed=false
			fi
		else
			B_TITLE=$(git log --pretty=%s -1 $B_HEAD)
			B_TAG=$(better_gdct $B_HEAD)
			printf "%s:\t${GREEN}%s %-23s\t%s${NC}\n" "$(branch2name ${BRANCH})" "${B_HEAD:0:13}" "(${B_TAG})" "${B_TITLE}"
			affect=true
		fi
	done

	if [ x$fixed != x"false" ]; then
		if [ x"$affect" == x"true" ]; then
			write_file $1 "[[${GREEN}All branch FIXED${NC}]]"
		else
			write_file $1 "[[${YELLOW}All branch not affected${NC}]]"
		fi
	fi

	echo

	return 0
}

check_branch
[ $? != 0 ] && exit -1;

[ x"$NEED_PULL" == x"true" ] && pull_branch

if [ -z "$FILENAME" ]; then
	CVES="$@"
else
	CVES=$(cat $FILENAME)
fi

for CVE in $CVES
do
	do_check $CVE
	if [ $? != 0 ]; then
		handle_local_fix $CVE
	fi
done
