package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`\P`!````Q!T```````!```````````"B````
M````!0```$``.``+`$``&@`9``8````$````0`````````!``````````$``
M````````:`(```````!H`@````````@``````````P````0```#,`@``````
M`,P"````````S`(````````A`````````"$``````````0`````````#``!P
M!````(:@`````````````````````````````'0````````````````````!
M``````````$````%````````````````````````````````````G)D`````
M``"<F0`````````0`````````0````8```"`FP```````("K````````@*L`
M``````#0!````````/@$`````````!`````````"````!@```)B;````````
MF*L```````"8JP```````"`"````````(`(````````(``````````0````$
M````J`(```````"H`@```````*@"````````)``````````D``````````0`
M````````!`````0```!\F0```````'R9````````?)D````````@````````
M`"``````````!`````````!0Y71D!````""1````````()$````````@D0``
M`````.P`````````[``````````$`````````%'E=&0&````````````````
M`````````````````````````````````````````!``````````4N5T9`0`
M``"`FP```````("K````````@*L```````"`!````````(`$`````````0``
M```````$````%`````,```!'3E4`9O0'=QR4X!_N_K7\2>W=.^SX5%HO;&EB
M+VQD+6QI;G5X+7)I<V-V-C0M;'`V-&0N<V\N,0`````[````1P````0````(
M````()`0`05`!,0H`@$$@(8$A(*)``()(`0(B(`00;H``HA'````2````$D`
M``!+````````````````````3`````````!-````3P```%``````````````
M`%$```!2`````````````````````````%,```!4````50````````!6````
M5P``````````````6`````````!9````````````````````6@```%L`````
M````70````````!>`````````%\`````````````````````````````````
M``!@````````````````````80````````!B````````````````````9```
M`!E^"H=_J9R;:@E#UC_/TFPA_^0.#3J7"W)0G]C#Y3\LB0=PT]EQ6!RM2^/`
M:QJ>4-5DU'R93\.;+\6;MC-BV^W_8?)A:W^:?+OCDGRS`8MO0OK7AT-%U>QG
MK\QF%8PES*/@^TG/WD<MXT?&`>"G`@IOQ=\?^Z%L8```````````````````
M`````````````````````````P`+`-`;``````````````````#G`@``$0``
M``````````````````````"E`@``$@````````````````````````!]`0``
M$@````````````````````````#%````$@`````````````````````````,
M`@``$@````````````````````````",`@``$@``````````````````````
M``#)`0``$@````````````````````````#-````$@``````````````````
M``````"%`@``$@`````````````````````````!`0``$@``````````````
M``````````!(`@``$@````````````````````````#6`0``$@``````````
M``````````````"]````$0````````````````````````#R````$@``````
M``````````````````"P`0``$@````````````````````````#7`@``$@``
M``````````````````````#\`0``$@````````````````````````"7````
M$@````````````````````````"A`0``$@`````````````````````````I
M`0``$@`````````````````````````9`0``$@``````````````````````
M``!8````$@`````````````````````````!````(```````````````````
M``````![````$@````````````````````````"]`@``$@``````````````
M```````````=````(`````````````````````````#K````$@``````````
M```````````````@`0``$@````````````````````````"O`@``$@``````
M``````````````````"3`0``$@````````````````````````"[````$0``
M``````````````````````"$````%@````````````````````````!;`0``
M$@`````````````````````````O`0``$@````````````````````````"$
M`0``$@````````````````````````#4````$@``````````````````````
M``!;`@``$@````````````````````````#"`0``$@``````````````````
M``````!*`0``$@````````````````````````#(`@``$@``````````````
M``````````"U`0``$@````````````````````````"L````$@``````````
M``````````````"J`@``$@````````````````````````!O`0``$@``````
M``````````````````#0`0``$@````````````````````````"7`@``$@``
M``````````````````````!V`0``$@````````````````````````"E````
M$@````````````````````````":`0``$0`````````````````````````M
M`@``$@````````````````````````"+`0``$@``````````````````````
M``#J````$@````````````````````````!)````(@``````````````````
M```````8`@``$@````````````````````````#B````$@``````````````
M``````````"]`0``$@````````````````````````#@`@``$@``````````
M``````````````!Y`@``$@````````````````````````#G`0``$@``````
M``````````````````"J`P``$@````````````````````````!L`@``$0``
M```````````````````````+`0``$@`````````````````````````_`0``
M$@`````````````````````````Y`@``$@`````````````````````````W
M````$@````````````````````````#Y````$@``````````````````````
M``!B`0``$@````````````````````````!J````$@``````````````````
M``````#U`0``$@````````````````````````#V`P``$`#Q_P"X````````
M``````````"8`P``$@`+`/Q!````````*`````````#:`P``$``5``"P````
M``````````````#G`P``$``5``"P``````````````````!K!```$@`+`(PX
M````````"`(```````#<`P``(``5``"P```````````````````3!```$``5
M`%"P```````````````````C!```$@`+`#(W````````/`````````!X!```
M$0`5`$BP````````"``````````K!```$``6`%"P``````````````````#+
M`P``$0`,`*A/````````!`````````".`P``$@`+`,Q-````````O@$`````
M``!(`P``$@`+`%Y)````````)`(```````""!```$@`+`$)`````````7@``
M```````J`P``$@`+`'PX````````$`````````#@`P``$@`+`,0=````````
M(@````````"_`P``$``6`'BP``````````````````!$````$@`+`-`;````
M````]`$```````!`!```$``6`'BP``````````````````!>`P``$@`+`.`_
M````````8@`````````(!```$@`+`+),````````&@$```````!2!```$``5
M`%"P``````````````````""`P``$@`+`,@Z````````/@$````````W!```
M$@`+`%),````````8`````````"*!```$@`+``8\````````V@,```````!%
M!```$@`+`"!#````````/@8````````[`P``$@`+`)0Z````````-```````
M``!L`P``$@`+`&XW````````#@$```````"S`P``$@`+`()+````````T```
M``````!9!```$@`+`"1"````````_```````````7TE435]D97)E9VES=&5R
M5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`7U]L:6)C
M7W-T87)T7VUA:6X`7U]C>&%?9FEN86QI>F4`7U]R96=I<W1E<E]A=&9O<FL`
M7U]S=&%C:U]C:&M?9F%I;`!R96%D;&EN:P!03%]C=7)R96YT7V-O;G1E>'0`
M4&5R;%]S=E]S971P=@!M96UC:'(`4&5R;%]S=E]S971P=FX`7U]E;G9I<F]N
M`'-T<FYC;7``<W1R;&5N`%]?;65M;6]V95]C:&L`<F5A;&QO8P!L<W1A=#8T
M`&%C8V5S<P!O<&5N9&ER`')E861D:7(V-`!?7W-P<FEN=&9?8VAK`'5N;&EN
M:P!C;&]S961I<@!R;61I<@!B;V]T7T1Y;F%,;V%D97(`4&5R;%]N97=84P!0
M97)L7VYE=UA37V9L86=S`&=E='!I9`!?7VUE;6-P>5]C:&L`<W1R8VAR`'-T
M<F1U<`!S=')T;VL`<W1R8VUP`'-T<G)C:'(`;65M8W!Y`'-T9&5R<@!?7W9F
M<')I;G1F7V-H:P!E>&ET`&QS965K-C0`<F5A9`!M96UC;7``9V5T=6ED`&UK
M9&ER`%]?97)R;F]?;&]C871I;VX`7U]F<')I;G1F7V-H:P!O<&5N-C0`4&5R
M;%]G=E]F971C:'!V`%!E<FQ?9V5T7W-V`%!E<FQ?8W)O86M?;F]C;VYT97AT
M`%!E<FQ?;6=?<V5T`%!E<FQ?<WES7VEN:70S`%!E<FQ?871F;W)K7W5N;&]C
M:P!097)L7V%T9F]R:U]L;V-K`%!,7V1O7W5N9'5M<`!097)L7W-A9F5S>7-M
M86QL;V,`<&5R;%]P87)S90!P97)L7V1E<W1R=6-T`'!E<FQ?9G)E90!097)L
M7W-Y<U]T97)M`'!E<FQ?86QL;V,`<&5R;%]C;VYS=')U8W0`<&5R;%]R=6X`
M;65M<V5T`%]?<W1A8VM?8VAK7V=U87)D`&QI8G!E<FPN<V\`;&EB8RYS;RXV
M`&QD+6QI;G5X+7)I<V-V-C0M;'`V-&0N<V\N,0!P87)?8W5R<F5N=%]E>&5C
M`'!A<E]B87-E;F%M90!84U]);G1E<FYA;'-?4$%27T)/3U0`<&%R7V5N=E]C
M;&5A;@!P87)?8W5R<F5N=%]E>&5C7W!R;V,`<&%R7V1I<FYA;64`<VAA7V9I
M;F%L`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%]?
M0E-37T5.1%]?`%])3U]S=&1I;E]U<V5D`%]?9&%T85]S=&%R=`!?7T1!5$%?
M0D5'24Y?7P!?7V=L;V)A;%]P;VEN=&5R)`!S:&%?=7!D871E`%]?4T1!5$%?
M0D5'24Y?7P!X<U]I;FET`%]?8G-S7W-T87)T`'-H85]I;FET`%]E;F0`<&%R
M7VUK=&UP9&ER`%]E9&%T80!P87)?<V5T=7!?;&EB<&%T:`!P87)?9FEN9'!R
M;V<`;7E?<&%R7W!L`'!A<E]D:64`<&%R7VEN:71?96YV`$=,24)#7S(N,S0`
M1TQ)0D-?,BXS,P!'3$E"0U\R+C(W`"]U<W(O;&EB+W!E<FPU+S4N-#(O8V]R
M95]P97)L+T-/4D4````````"``$``P`#``$``0`#``,``P`#``$``P`#``,`
M`P`!``$``0`#``,``P`#``$``P`!``$`!``#``$``P`#``$``P`!``,``P`!
M``,``0`!``,``0`#``,``P`!``,``P`#``$``P`$``,``0`#``,``P`!``,`
M`P`!``,``0`!``4``P`#``,``P`!``$``0`!``$``0`!``$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0````$``P`$`P``
M$````$````"TD98&```%`)<$```0````LY&6!@``!`"B!```$````(>1E@8`
M``,`K00````````!``$`#@,``!``````````AY&6!@```@"M!````````("K
M`````````P````````#F'0```````(BK`````````P````````"F'@``````
M`)"K`````````P````````!H'@```````&"O`````````P````````"*3P``
M`````/"O`````````P````````#0&P````````"P`````````P``````````
ML````````!BP`````````P`````````(40```````""P`````````P``````
M``#@D````````"BP`````````P````````#HD````````#"P`````````P``
M``````#PD````````#BP`````````P````````#XD````````$BP````````
M`P````````!P4P```````."M`````````@````(``````````````.BM````
M`````@````,``````````````/"M`````````@````0``````````````/BM
M`````````@````4```````````````"N`````````@````8`````````````
M``BN`````````@````<``````````````!"N`````````@````@`````````
M`````!BN`````````@````D``````````````""N`````````@````H`````
M`````````"BN`````````@````L``````````````#"N`````````@````P`
M`````````````#BN`````````@````T``````````````$"N`````````@``
M``X``````````````$BN`````````@````\``````````````%"N````````
M`@```!```````````````%BN`````````@```!$``````````````&"N````
M`````@```!(``````````````&BN`````````@```!,``````````````'"N
M`````````@```!0``````````````'BN`````````@```!4`````````````
M`("N`````````@```!8``````````````(BN`````````@```!@`````````
M`````)"N`````````@```!D``````````````)BN`````````@```!H`````
M`````````*"N`````````@```!L``````````````*BN`````````@```!P`
M`````````````+"N`````````@```!T``````````````+BN`````````@``
M`!X``````````````,"N`````````@```!\``````````````,BN````````
M"P```"$``````````````-"N`````````@```"(``````````````-BN````
M`````@```",``````````````."N`````````@```"0``````````````.BN
M`````````@```"4``````````````/"N`````````@```"8`````````````
M`/BN`````````@```"<```````````````"O`````````@```"@`````````
M``````BO`````````@```"D``````````````!"O`````````@```"H`````
M`````````!BO`````````@```"L``````````````""O`````````@```"P`
M`````````````"BO`````````@```"T``````````````#"O`````````@``
M`"X``````````````#BO`````````@```"\``````````````$"O````````
M`@```#```````````````$BO`````````@```#$``````````````%"O````
M`````@```#(``````````````%BO`````````@```#,``````````````&BO
M`````````@```#0``````````````'"O`````````@```#4`````````````
M`'BO`````````@```#8``````````````("O`````````@```#<`````````
M`````(BO`````````@```#@``````````````)"O`````````@```#D`````
M`````````)BO`````````@```#H``````````````*"O`````````@```#L`
M`````````````*BO`````````@```#P``````````````+"O`````````@``
M`#T``````````````+BO`````````@```#X``````````````,"O````````
M`@```#\``````````````,BO`````````@```$```````````````-"O````
M`````@```$$``````````````-BO`````````@```$,``````````````."O
M`````````@```$0``````````````.BO`````````@```$4`````````````
M`/BO`````````@```$8``````````````,BM````````!0```!<`````````
M`````-"M````````!0```$(`````````````````````````EY,``#,#PT$#
MOH,B$P-#_9."@R(34Q,`@[*"`&<`#@`7G@```SZ.(6<##@`3````%YX```,^
M#B%G`PX`$P```!UQANRBZ,K@@!"FY$[\%YD```,Y*2"#-PD`(S3T_(%'(SBT
M^B,TQ/HC+J3Z$P:$^I,%!/L3!<3[%Y,```,S@SP"DQ>6```#-N8ALH47E0``
M`S5%+1>3```#,\,S`I.7EP``@[>G.(/'!P!CAP<0EY<``(.WIRB7E```DX0D
M09*7F&,#)L3[%Y,```,SHS2#1Z=+&P5&``X%D^<W`",-]TH-)B,PQ/H"DX,U
M!/L7EP```S?'/`,HQ/N#N`4`JH:7-P``DX>G;1<U```3!:5LG.Z4Y".P%@&(
MYICJA4<#-@3Z8]T'#Z$%#2@11XAADQ<W`+:7!2>(XZ$%XQD'_PX'B&PVER,P
M!P"7)0``DX4%GP%'%Y,```,S@Q("DUG)B&P7DP```S.C)`*3DP5$_*J)%S4`
M`!,%I3DC,+3Z[Q!@3X,U!/HUS8AL%Y,```,S(PT"DQ>3```#,X,9`I,#-X3\
M@S<)`+F/`4?1X^9@1F2F9`9I3H7B>25A@H`7DP```S,#%0*3EY0``).$9#"(
M["G)%Y,```,SHQH"DY>7``"#MP<6,X='`!AC$P?W?Z,+!Q+!O8AL%Y,```,S
M8PT"DX6WB&@!Q>\@$%^]OQ<U```3!84H[Q``1HCH/=7MMQ%'!;<%11>3```#
M,P,*`I,7DP```S/C(@*3[P`@`JJ'%Y4```,U92*"93``$W$!_X%&`4<*B._P
M']X"D)>A``"3@:&A@H`!`$$1(N`&Y``(%Y4``!,%9267EP``DX?G)&.+IP"7
MEP``@[>G!XG'`F2B8$$!@H>B8`)D00&"@!>5```3!84BEY4``).%!2*)C4$1
MD]<U0/V1(N`&Y``(OI6%A9')EY<``(.W)P6)QP)DHF!!`8*'HF`"9$$!@H"7
MEP``@\>''I7K01$BX`;D``B7EP``@[?G#Y''%Y4```,UQ1>"E^_P7_:B8`)D
MA4<7EP``(PSW&D$!@H""@$$1(N`&Y``(`F2B8$$!E;\3`0'1(S"!+H`=(S1Q
M*R,T$2XC/)$L(S@A+2,T,2TC,$$M(SQ1*R,X82LC,($K(SR1*2,XH2F7FP``
M@[M+[Q,'A-%!;Y,(A0.3`H4'@[<+`",\]/B!1SJ($P\/\+</_P`#M@@`H0A!
M")-5!@*3=O8/DQZ&`)-7A@(37H8`$_/U#Y-3A@/B!K/^_@&S]^<!L^;6`3-^
M[@%B`Z(%L^=W`+/FQ@$;5H8!L_7U`;/G9P#1CLV/(SC8_B,\^/[CDQ+[@S,$
MV(,RA-B#/P39`SN$TH,Z!-,#.H33@SD$U`,YA-2#-`35GHP6@WZ,6HW6B%*(
MSH5*AB:.$P\'(!&@-HXT8QQCXHXA![6/L\>G`;/'EP$3G!<`_8,S;/P``AP3
M7`P"(SR'!T:-FHS"B':#+HBRA7*&XQ7G_0,S!0`0:0QM@S:$T0,XA0"#.`4"
MMX>"6I.'EYD35[,!$QY3`+Z6,V[N`#-'M@#&EC-W!P$MCS:>.IZ3'N@!$U<H
M`+/N[@`3'.,!$U<C`#-L[``"'A->#@(SQXX!LP;[``,_!-(S=\<!LI8S1]<!
M-I>S1M8!/I^S]F8`+I^QCKZ:]IKZEI->O@$3'UX`LV[?`?:6@A:!DC,/^@"3
MGE8`$]JV`;/N3@%VEY,>[@$37BX`L^[.`3-.W`$S?MX``A<!DS-.C@%6GA,:
M5P"36K<!,VI:`5*>$YKF`8F",VK:`+/&3@$"'A->#@+YCF*?L\;6`9,:7@`3
M6[X!^I:SZFH!$Q_G`0F#,V_O`-:6,T?J`8(6@9*^F3-WQP'.GC-'1P&3F58`
MD]JV`7:7L^Y9`7:7DQ[N`1->+@"S[LX!,T[?`0(7`9,^F3-^W@!2F9,95P`S
M3NX!$UJW`4J>,^E)`9.9Y@&)@K/IV0"SQCX!2IZ^E/F.`AX37@X")I^SQM8!
MDU2^`?J6$Q]>`#-OGP#ZEH,TA-43'^<!"8,S;^\`,\?I`;Z4,W?'`8(6@9*F
MGC-'-P&3U+8!=I>3GE8`L^Z>`':7@S0$UI,>[@$37BX`L^[.`3-)WP$SCO0`
M,WG9``(7`9-.GC-)Z0&35+<!<ID3'E<`@SF$UC-NG@"3E.8!B8+5C'*9,\Z>
M`+.&^0`S?NX``AD360D"^I8S3MX!$U^Y`3:>DQ99`(,Y!->SYN8!$Q_G`0F#
M,V_O`#:>L\;D`3.'^0"S]B8!`AX37@X"=I>ECI->O@&ZEA,77@"#.837,V?7
M`9,>Z0$362D`L^XN`;J6@A8S1]\!@9(SB?D`,W?'`<J4,T?G`1.95@"3V;8!
M)I>S9#D!$QGN`1->+@`S;LD!)I>3%`<",\?.`8&0OI-UCQZ?$]FT`9.35``S
M1]<!>I<S[R,!>I<3G^8!B8*S9M\`DQ,'`C-'W@"^DF6/D],#`I:>,T?'`1/?
MLP%VEY.>4P"S[NX!=I>3GN0!B8"S[IX`DQ('`C/'U@&^GS-W=P"3T@("?IXU
MCQ/?L@%REQ.>4@`S;NX!@S^$V1.?XP&3TR,`,V]_`'*7$QX'`C/'[@&^GS-W
M5P`37@X"_I8S1]<!DU^^`3:7DQ9>`(,S!-JSYO8!DY_B`9/2(@"S[U\`-I>3
M%@<",T?_`;."\P`S=\<!@9*6GC-'YP&3TK8!=I>3GE8`L^Y>`':7@S*$VI,>
M[@$37BX`L^[.`3/.WP&^DC-^W@`"%P&3%I\S3OX!DU*W`7J>$Q]7`#-O7P!Z
MG@,_!-N3DN8!B8*SXM(`LP;_`+/'7@"3$PX"^8^3TP,"_I:SQ]<!MI<3W[,!
MDY93`+/FY@$#/X3;OI:3%.<!M_?9;@F#DX<7NMF,,\>2`#Z?DY\&`I/?#P(S
M3L<!,P??`3J>D]Z_`1.77P`S9]<!.IX#-P3<D]XC`!.?XP$S;]\!L\[D`9,3
M#@(^EY/3`P*SQMX`%I<VEY/>LP&3EE,`L^;6`;J6`S>$W).>[P&3WR\`L^[^
M`;-/WP&3D@8"/I>3T@(",\[/`2:7<I>3W[(!$YY2`#-N_@$ZG@,W!-V3U",`
MDY_C`;/OGP"SQ/X!DQ,.`CZ7D],#`J6.>I<VEQ/?LP&3EE,`L^;F`;J6`S>$
MW9/4(@`3G^(!,V^?`+/$[P&3D@8"/I>3T@(",\[$`7:7<I>3WK(!$YY2`#-N
MW@$ZG@,W!-Z3U",`DY[C`;/NG@"S1-\!DQ,.`CZ7D],#`J6.?I<VEY/?LP&3
MEE,`L^;V`;J6`S>$WI/4(@"3G^(!L^^?`+/$_@&3D@8"/I>3T@(",\[$`7J7
M<I<3W[(!$YY2`#-N[@$ZG@,W!-^3U",`$Y_C`3-OGP"SQ.\!DQ,.`CZ7D],#
M`J6.=I<VEY/>LP&3EE,`L^;6`;J6`S>$WY/4(@"3GN(!L^Z>`+-$WP&3D@8"
M/I>3T@(",\[$`7Z7<I>3W[(!$YY2`#-N_@$ZG@,W!."3U",`DY_C`;/OGP"S
MQ/X!DQ,.`CZ7D],#`J6.>I<VEQ/?LP&3EE,`L^;F`;J6`S>$X)/4(@`3G^(!
M,V^?`+/$[P&3D@8"/I>3T@(",\[$`7:7<I>3WK(!$YY2`#-NW@$ZG@,W!.&3
MU",`DY[C`;/NG@"S1-\!DQ,.`CZ7D],#`J6.?I<VEY/?LP&3EE,`L^;V`;J6
M`S>$X9/4(@"3G^(!L^^?`+/$_@&3D@8"/I>3T@(",\[$`7J7<I<3W[(!$YY2
M`#-N[@$ZG@,W!.*3U",`$Y_C`3-OGP"SQ.\!DQ,.`CZ7D],#`J6.=I<VEY/>
MLP&3EE,`L^;6`;J6`S>$XI/4(@"3GN(!L^Z>`+-$WP&3D@8"/I>3T@(",\[$
M`7Z7<I>3W[(!$YY2`#-N_@$ZG@,W!..3G^,!D],C`+/O?P"SP_X!DQ0.`CZ7
M@9"SQM,`>I<VEQ/?M`&3EE0`L^;F`;J6`S>$XQ.?X@&3TB(`,V]?`+/"[P&3
MDP8"/I>3TP,",\["`7:7<I>3WK,!$YY3`#-NW@&S#L<!`S<$Y!/>)`"3DN0!
ML^+"`3-.7P"3E`X"/I>!D+-&W@!^ES:7$]ZT`9.65`"SYL8!,P[7`(,VA.03
MUR,`DY_C`;/O[P`SQ_(!DQ,.`KZ6D],#`C-'UP'ZEI/>LP&ZEA.74P`S9]<!
MNI8#-P3E$Y_D`8F`,V^?`+/.[P&ZEX(6@9(SQ\X!EI<3WK8!NI<3EU8`,V?'
M`;J7@S*$Y1.>!P*W]\8CDY[C`8H'D],C`).'Q\VS[GX`,V??`;Z2EI]UC[-R
MWP$37@X",V=7`'Z7DU*^`9,?7@"S[U\`@S($YGZ7DY_F`8F"L^_?`+Z2DQ8'
M`C/G_@$6GS-WQP&S\OX!@9(S9U<`>I>3TK8!$Y]6`#-O7P"#,H3F>I<3'^X!
M$UXN`#-OSP&^DA,>!P(SY^\!EIYUC[/R[P$37@X",V=7`':7DU*^`9,>7@"S
M[EX`@S($YW:7DY[F`8F"L^[>`+Z2DQ8'`C-GWP&6GS-WQP&S<M\!@9(S9U<`
M?I>3TK8!DY]6`+/O7P"#,H3G?I>3'^X!$UXN`+/OSP&^DA,>!P(SY_X!%I]U
MC[/R_@$37@X",V=7`'J7DU*^`1,?7@`S;U\`@S($Z'J7$Y_F`8F",V_?`+Z2
MDQ8'`C/G[P&6GC-WQP&S\N\!@9(S9U<`=I>3TK8!DYY6`+/N7@"#,H3H=I>3
M'NX!$UXN`+/NS@&^DA,>!P(S9]\!EI]UC[-RWP$37@X",V=7`'Z7DU*^`9,?
M7@"S[U\`@S($Z7Z7DY_F`8F"L^_?`+Z2DQ8'`C/G_@$6GS-WQP&S\OX!@9(S
M9U<`>I>3TK8!$Y]6`#-O7P"#,H3I>I<3'^X!$UXN`#-OSP&^DA,>!P(SY^\!
MEIYUC[/R[P$37@X",V=7`':7DU*^`9,>7@"S[EX`@S($ZG:7DY[F`8F"L^[>
M`+Z2DQ8'`C-GWP&6GS-WQP&S<M\!@9(S9U<`?I>3TK8!DY]6`+/O7P"#,H3J
M?I>3'^X!$UXN`+/OSP&^DA,>!P(SY_X!%I]UC[/R_@$37@X",V=7`'J7DU*^
M`1,?7@`S;U\`@S($ZWJ7$Y_F`8F",V_?`+Z2DQ8'`C/G[P&6GC-WQP&S\N\!
M@9(S9U<`=I>3TK8!DYY6`+/N7@"#,H3K=I>3'NX!$UXN`+/NS@&^DA,>!P(S
M9]\!EI]UC[-RWP$37@X",V=7`'Z7DU*^`9,?7@"S[U\`@S($['Z7DY_F`8F"
ML^_?`+Z2DQ8'`C/G_@$6GS-WQP&S\OX!@9(S9U<`>I>3TK8!$Y]6`#-O7P"#
M,H3L>I<3'^X!$UXN`#-OSP&^DA,>!P(SY^\!EIYUC[/R[P$37@X",V=7`':7
MDU*^`9,>7@"S[EX`@S($[7:7DY[F`8F"L^[>`+Z2DQ8'`C-GWP&6GS-WQP&S
M<M\!@9(S9U<`?I>3TK8!DY]6`+/O7P"#,H3M?I>3'^X!$UXN`+/OSP&^DA,>
M!P(SY_X!%I]UC[/R_@$37@X",V=7`'J7DU*^`1,?7@`S;U\`@S($[GJ7$Y_F
M`8F",V_?`+Z2DQ8'`C/G[P&6GC-WQP&S\N\!@9(S9U<`=I>3TK8!DYY6`+/N
M7@"#,H3N=I>3'NX!$UXN`+/NS@$S;M\!OI*6GS-^W@"S<M\!`A<!DS-N7@!^
MGI-2MP&3'U<`L^]?`'Z>@S\$[Y.3Y@&)@K/CTP"SAO\`L^=^`+/_?@#ZEOF/
M$Q\.`A-?#P*SY_<!MI>37[\!DQ9?`+/F]@&^EH,_A.^WMY@R$QGG`8H'"8.3
MAV<=,VGI`+Z?DY(&`C/'(P&3T@(",T?'`3..WP&S#L<!$]ZR`1.74@`S9\<!
MNIX#-P3PDQ3O`1-?+P"SY.0!DY\.`C-.F0`^EY/?#P(>ES-.W@`ZGI/6OP$3
MEU\`58\ZG@,WA/"3UB(`$Y_B`3-OWP"SQN0!DQ(.`CZ7D]("`K/&U@%*ES:7
MD]ZR`9.64@"SYM8!NI8#-P3QDY[O`9/?+P"S[OX!LT_?`9.3!@(^EY/3`P(S
MSL\!)I=REY/?LP$3GE,`,V[^`3J>`S>$\9/4(@"3G^(!L^^?`+/$_@&3$@X"
M/I>3T@("I8YZES:7$]^R`9.64@"SYN8!NI8#-P3RD]0C`!.?XP$S;Y\`L\3O
M`9.3!@(^EY/3`P(SSL0!=I=REY/>LP$3GE,`,V[>`3J>`S>$\I/4(@"3GN(!
ML^Z>`+-$WP&3$@X"/I>3T@("I8Y^ES:7D]^R`9.64@"SYO8!NI8#-P3SD]0C
M`).?XP&S[Y\`L\3^`9.3!@(^EY/3`P(SSL0!>I=REQ/?LP$3GE,`,V[N`3J>
M`S>$\Y/4(@`3G^(!,V^?`+/$[P&3$@X"/I>3T@("I8YVES:7D]ZR`9.64@"S
MYM8!NI8#-P3TD]0C`).>XP&S[IX`LT3?`9.3!@(^EY/3`P(SSL0!?I=REY/?
MLP$3GE,`,V[^`3J>`S>$])/4(@"3G^(!L^^?`+/$_@&3$@X"/I>3T@("I8YZ
MES:7$]^R`9.64@"SYN8!NI8#-P3UD]0C`!.?XP$S;Y\`L\3O`9.3!@(^EY/3
M`P(SSL0!=I=REY/>LP$3GE,`,V[>`3J>`S>$]9/4(@"3GN(!L^Z>`+-$WP&3
M$@X"/I>3T@("I8Y^ES:7D]^R`9.64@"SYO8!NI8#-P3VD]0C`).?XP&S[Y\`
ML\3^`9.3!@(^EY/3`P(SSL0!>I=REQ/?LP$3GE,`,V[N`3J>`S>$]I/4(@`3
MG^(!,V^?`+/$[P&3$@X"/I>3T@("I8YVES:7D]ZR`9.64@"SYM8!NI8#-P3W
MD]0C`).>XP&S[IX`LT3?`9.3!@(^EY/3`P(SSL0!?I=REY/?LP$3GE,`,V[^
M`3J>`S>$]Y/?(@"3E.(!L^3T`;//G@"3$@X"/I>3T@("L\;?`'J7-I<3W[(!
MDY92`+/FY@&ZE@,W!/@3WR,`DY_C`;/O[P$SS_0!DY,&`CZ7D],#`C-.SP%V
MEW*7D]ZS`1.>4P`S;MX!LP['`0,WA/@3WB(`$Y_B`3-OSP&3D@X",\[O`3Z7
MD]("`K-&W@`FEQ.>4@"3U+(!-I>#-@3Y,VZ>`'*7$Y[C`9/3(P`S;GX`MI<"
M%P&3FI>S1L\!L\;6`1,35P"37K<!_I>VES-CTP&3EN(!D](B`+/F5@":ESJ8
M^I@SA\8`\I6"%X&3$Y8(`I,6"`*"%0(7@9(!DH&1`9,<X13E$/$,[8,VA/F#
MMPL`M8^!1ACIA>^#,($N`S0!+H,T@2T#.0$M@SF!+`,Z`2R#.H$K`SL!*X,[
M@2H#/`$J@SR!*0,]`2D3`0$O@H`7@P```S-C[P*377&BX(;D)OQ*^$[T4O"`
M"(5B,P%10",P`4#]=Y.'A_P%9I>$``"#M"3,JHDNBGT6LX6'`!<E```3!07H
M%X,```,S@]6<8",T]/R!1P*3FP8%`&-?T`"3!P2`DX<'_9.'!X`;A_;_NI>#
MQX?_H>,5[Y>'``"#M\?54H;.A9*7B&,7@P```S/#S@*3`S>$_)Q@N8\!1[WC
MA6(6D:9@!F3B=$)YHGD">F%A@H"ZAOUWDX>'_#.)AP#]=Y.'A_NBETJ%-H:3
M!?`"%X,```,S8]B4XP*36=V7AP``@[>'SWUW$P>'^R*7DI<48XAC2H;.A1>#
M```#,^/2`I-)OQ>#```#,R/?`I-=<:+@AN0F_%KH@`B7AP``@[='PP.[!P!C
M"PL(3O2#1P4`@<>3AS?\N>^!28,T"P#)P$KX4O!6[%[D8N`JBBZ,6HF7B@``
M@[IJNY,+T`-.AM*%)H6"FK.'-`$)Y8/'!P!CCG<#@S2)`"$)]?!">:)Y`GKB
M:J)K`FRF8`9D0FLFA>)T86&"@*J)@\<9`(4)$X<W_)'#=?N[B:E`2;\S"6E!
M$UDY0(4)(R`L`<Z4P;>!1.&WHGG1MUEQHO"F[,KHSN2&]-+@5OR`&)>)``"#
MN8FQ`\<%`(.W"0`C//3Z@4>3!]`#KH0JB6,4]P"3A!4`%X,```,SH[(FA0*3
M*HJ3!43[2H7O\/_OJHI]R1>#```#,^.P`I-C=D4-`R9$^Y>*``"#NDJR#@:#
M1PD`8X4'&).'-_QC@0<82H6#1Q4`!06!QY.'-_SU^SL%)4&#MPH`"0I2E1>#
M```#,^.L,XK'`",TQ/H"DR,PJ@`AS8.W"@`#-H3Z`T<)`+*7G&.3!C?\&P8'
M`".`YP"-P@7"$P49`"&@DX8V_)'*`T<%``4%A0<C@.<`FP8'`&WW$P?0`R.`
MYP"%!P/'!`"%!(4'HX_G_G7[`S>$^X.W"0"YCP%'Y>NF<`9TYF1&::9I!FKB
M>F5A@H`#QP0`A02%"J./ZOYQVP/'!`"%!(4*HX_J_GWST;>7B@``@[HJI`.X
M"@"#-P@`S<="AX%'%&<A!SZ&A2?E_IL&-@".!A>'```#)T?#->,%1S:%%X,`
M``,SXYXC,/3Z(SP$^1>&```C*^;`(S34^@*3)=F#-P3Z@S:$^H,UA/D3EC<`
M%X,```,S8Z@C-,3Z(SRD^`*3`S>$^8,W!/H#-H3Z([#J`(4GC@<^ER,P!P!A
MO;:%0H47@P```S-#KR,T]/H"DR.PJ@`JAP'=@S>$^A.6-P#)OP%%4;W!1JVW
M%X,```,SP[("DW%Q(O$F[4KI`!D&]9>$``"#M.20DP6$]1>#```#,R.IG&`C
M//3\@4<JB0*3#>6#)X3VO6819_6/8X7G`"EG8YWG`$J%B447@P```S/#DP*3
M$S45`!&@`44#-X3]G&"YCP%'F>>J<`ITZF1*:4UA@H`7@P```S,CJP*3%7&B
MZ4[]ANV`$9>)``"#N6F)@[<)`",\]/J!1QG%IN6#1P4`JH29[ZYD`S>$^X.W
M"0"YCP%'8YP'$NY@3F3J>2UA@H#*X1>#```#,R.E`I,JB6,'!1)2^1>*```#
M.@J)2H4"FG'-`T<U`9,'-0$;!R?]FP8'`!GC@\87`/72`T<U`1L')_V;!@<`
M`>L#QQ<`&P<G_9L&!P!AQ_G25O47@P```S.C@R:%(S3T\@*3@S>$\JJ*%X,`
M``,S0X(^A2,P]/("DSN&J@`))C*%%X,```,S0X$C-,3R`I.#-P3R`S:$\B:'
MER8``).&QIF)11>#```#,R.9JHH"DY,%A/-6A1>#```#,X.&`I/]5V,*]0"#
M)X3TD483U\=`/8MC!=<$5H47@P```S/C@0*35H47@P```S,CBP*32H6J>@*:
M!?5*A1>#```#,^."`I,FA1=S```#,Z-^`I.N9`YI2GKQM8/&)P`=OU:%[_"?
MZ<&WIN7*X5+Y5O47@P```S,CDP*3KF0.:5VU`1$FY)=T``"#M.1_(N@&[``0
MLX=$`(ACEV8``).&IJ(7=@```S8F?Y<E``"3A:6-%X,```,S(XT"DY*40F2(
M8.)@HF07)P``$P?GQ+J&@4<7%@``$P:&))<E``"3A:6,%W,```,SHWT%80*#
MEW<``(.W9WE!$2+@!N0`")*7`F2B8(ACEV8``).&YIL7=@```S9F>)<E``"3
MA>6&%X,```,S8X9!`0*#'7&BZ(;LIN3*X$[\4OB`$(5B,P%10",P`4`S`5%`
M(S`!0)=Y``"#N0EE%W,```,S@W.#MPD`(S3T_(%'`I/]=Y.'!_RS!/0`!68J
MAP4&)H67)P``DX<GA)<F``"3AB:$B447<P```S,#?P*3^7>3AX?[,PGT`":%
MRH4%9A=S```#,V-J`I.;!`4`@4=CR@0$^7>;AA0`DX<'^J*7-H47<P```S-#
M893C`I.JAQ7)^7>3AP?ZHI>48WEW$P>'^B*7)H;*A1=S```#,^-Z".,"D_EW
MDX>'^J*7G&.^E".`!`"#-H3\`[<)`#6/@48![XEB%I'F8$9DIF0&:>)Y0GH^
MA25A@H`7<P```S-C=P*301$BX`;D``@"9*)@00'5M5%QHO&&]:;MRNG.Y=+A
M5OU:^5[U8O%F[8`9A6(S`5%`(S`!0/UWDX='\1=\```#/*Q2*HJNA!<5```3
M!>5ULP7T`(,W#``C//3X@4?O\)^2*HF3!?`"4H47<P```S-#9`*3(<$7%0``
M$P4%=-*%[_#?FU*%`S>$^8,W#`"YCP%'8Y\'%H5B%I&N<`YT[F1.::YI#FKJ
M>DI[JGL*?.IL;6&"@":%%W,```,S`V$"DY<5``"3A65P%W,```,SXTH"DZJ$
M3=']=Y.'A_&S"_0`@\<$`'UW$P>'^;,*Y`"7>0``@[D)2P5KO<=C"@D`IH5*
MA1=S```#,T-6`I,5S2:%@IF3!_7_II>JC),&\`)CZ/0`*:@C@`<`_1=CAO0`
M`\<'`.,)U_XFA8*9JHQ2A8*99I4)!6-H9067%0``DX7E9P%%%W,```,S0T("
MDZJ$`=V#QP0`V?MC#`D`@T<)`)N')_T;AP<`F>,#1QD`<==2A85,@IEFE0D%
MEQ0``).$9&'C?&7[B447<P```S,#6R:'4HB7%P``DX?'8I<6``"3AL9B!696
MA0*3WH56A1=S```#,R-'`I-)\9,'!("3AP?ZDX<'@(.GA_BA1A/7QT`]B^,4
MU_:%15:%%W,```,S8SX"DSGYUH47%0``$P4%7._P_X-6A1=S```#,R-,`I.E
MO1=S```#,^-5`I,!$2+H)N0&[``0DP7P`A=S```#,V-,JH0"DQG!DP05`$)D
MXF`FA:)D%W,```,S0T@%80*#.7$B^`;\)O1*\(``A6(S`5%`(S`!0)=T``"#
MM"0PG&`C//3\@4=AS8-'!0#IR_UWDX>'_**7%W,```,SHS&(XP*3$P85``5G
MJH=C8\<._7:3AH;\HI:,8GUW$P>'_3,%Y``7<P```S-C.9SB`I/]=Y.'A_RB
MEYQC*H>3!O`"$X7W_SJ58VBG`!&H(P`%`'T58P;E`(-'!0#CB=?^_7>3AX?]
M?7>BEQ,'A_P^A2*7DP7P`A=S```#,^,^'.,"D_UWDX>'_**7G&,=R6/@IP;]
M=Y.'A_TS!?0`%W,```,S`SH"DP,WA/V<8+F/`4>AYX5B%I'B<$)THG0">2%A
M@H`#-X3]G&"YCP%'E>>%8A:1XG!"=*)T`GD7%0``$P6E0Q=S```#,Z,U(6$"
M@R,`!0!%MP%%1;\7<P```S/#/@*3.7$B^$KP3NP&_(``%WD```,YZ1R7>0``
M@[EI(H,W"0`C-/3\@4>3!43\%Q4``!,%A4+O\(_<'<4#)T3\@[<)``X'NI>8
M9YCC>=^8:Z$'F.-M_Y,%1/P7%0``$P7%/^_PS]EY^9=Y``"#N6D=DP5$_!<5
M```3!24_[_`OV!W%`R=$_(.W"0`.![J7F&>8XWG?F&NA!YCC;?^3!43\%Q4`
M`!,%93SO\&_5>?F7>0``@[D)&9,%1/P7%0``$P5%.^_PS],=Q0,G1/R#MPD`
M#@>ZEYAGF.-YWYAKH0>8XVW_DP5$_!<5```3!84X[_`/T7GYEWD``(.YJ123
M!43\%Q4``!,%Y3?O\&_/'<4#)T3\@[<)``X'NI>89YCC>=^8:Z$'F.-M_Y,%
M1/P7%0``$P4E->_PK\QY^9=Y``"#N4D0DP5$_!<5```3!04T[_`/RQW%`R=$
M_(.W"0`.![J7F&>8XWG?F&NA!YCC;?^3!43\%Q4``!,%13'O\$_(>?F7>0``
M@[GI"Y,%1/P7%0``$P6E,._PK\8=Q0,G1/R#MPD`#@>ZEYAGF.-YWYAKH0>8
MXVW_DP5$_!<5```3!>4M[_#OPWGYEWD``(.YB0>3!43\%Q4``!,%Q23O\$_"
M'<4#)T3\@[<)``X'NI>89YCC>=^8:Z$'F.-M_Y,%1/P7%0``$P4%(N_PC[]Y
M^9=Y``"#N2D#DP5$_!<5```3!>4H[_#OO1W%`R=$_(.W"0`.![J7F&>8XWG?
MF&NA!YCC;?^3!43\%Q4``!,%)2;O\"^[>?F7>0``@[G)_I,%1/P7%0``$P6%
M)>_PC[D=Q0,G1/R#MPD`#@>ZEYAGF.-YWYAKH0>8XVW_DP5$_!<5```3!<4B
M[_#/MGGYEWD``(.Y:?J3!43\%Q4``!,%)2+O\"^U'<4#)T3\@[<)``X'NI>8
M9YCC>=^8:Z$'F.-M_Y,%1/P7%0``$P5E'^_P;[)Y^9=Y``"#N0GVDP5$_!<5
M```3!444[_#/L!W%`R=$_(.W"0`.![J7F&>8XWG?F&NA!YCC;?^3!43\%Q4`
M`!,%A1'O\`^N>?F3!43\%Q4``!,%Y1KO\.^L`<FJA1<5```3!>47[_!/MY,%
M1/P7%0``$P5E&N_P[ZH!R:J%%Q4``!,%Y1KO\$^UDP5$_!<5```3!>4:[_#O
MJ!7-JH47%0``$P5E"N_P3[,#-X3\@S<)`+F/`4>IXT)TXG`">>)IEQ4``).%
M11H7%0``$P5%#B%A;_"/L),%1/P7%0``$P4E%^_P+Z1QT:J%%Q4``!,%)0[O
M\(^N5;\7<P```S-#`2;T`I-Y<2+P)NP`&`;TEW0``(.T9-^3!43]%Q4``!,%
M)0N<8",\]/R!1^_PKY\!S8-'!0`3A0?]$S45`).W%P!=C1-%%0`#-X3]G&"Y
MCP%'D>>B<`)TXF1%88*`%W,```,S`_L"DQUQ(N@&[``0%W,```,S8_`JC@,U
M`P`3`X0`#.00Z!3L'/08\",X!`,C/!0#EW<``(.W)]>:AB,P9/YRAA=S```#
M,R/?F&,C-.3^`4>)10*3$P7P#Q=S```#,Z/;`I-9<4%CHO"&]*;LRNC.Y-+@
M5OQ:^%[T8O!F[&KH@!@S`V%`A6(S`5%`(S`!0.,<8?Z7?```@[P,T8.W#``C
M//3X@4<)1H%%%W,```,SH^*JB@*3DPGU_Q/7^4-!D[.'Z0#"%\&3F8^SB?E`
M8\@)"L%WDX<'^3,)]``7?````SR,WWU:EWL``(.[Z^8=2P%&SH56A0*<8P5%
M"4%F(0;*A5:%@IL!)6,*10EC=:L&816S!*D`8^`D!Q=]```#/2W:(:#]%&/H
M)`4A1I=U``"3A07J)H4"G77UP7>3AP?YHI<SA?1`3I4#-X3Y@[<,`+F/`4>M
MX\%B%I&F<`9TYF1&::9I!FKB>D)[HGL"?.)L0FUE88*`P7>^F>.;^?9]5>&W
M`S>$^8.W#`"YCP%'A><7%0``$P7%]^_P_^8#-X3Y@[<,`+F/`4>9YQ<5```3
M!27W[_!?Y1=S```#,V/?`I-!$2+@!N0`"!=S```#,\/``I,7<P```S,CV@*3
M$<$(8:)@`F1!`8*`77&BX";\2OB`"(;DEW0``(.T!+LJB9,%1/P7%0``$P6E
M\IQ@(S3T_(%'[^`_^P'%@T<%`)WC`S>$_)Q@N8\!1\J%%Q4``!,%)?#1[P9D
MIF#B=$)Y86%O\,^#3O0C/*3Z%W,```,S0[E*A0*3`SB$^ZJ)%W,```,S([A"
MA2,X!/L"DS.&J0`)!C*%%W,```,S([<C/,3Z`I,#.`3[`S:$^TJ'EQ<``).'
MI]>7%@``DX8FV(E%%W,```,S@\ZJB0*3`S>$_)Q@N8\!1X7G!F2F8.)T0GG.
MA:)Y%Q4``!,%Q>9A86_@'_M.]!=S```#,\/-`I,7<P```S,CS0*3"7&7=P``
MDX=GSJ+ZRO(``\[NTNJ&_J;V`[B'`P._!P&#OH<!`[X'`@.SAP*#N`<#$PG$
MZ9=Y``"#N<FH*HH7%@``$P8FXI<6``"3AB;+%Q<``!,'I^&7%P``DX>GX2,P
M!/'*A1<5```3!<7)`[@)`",T!/D!2",X!.PC/`3J(SSD[2,PU.\C-,3O(SAD
M[B,\%.\C,,3L(S34[",TY.HC./3J[^#_XQ'%@T<%`*J$8YH'*M;FVN)>_F+Z
M9O87<P```S-#HP*3%W,```,SH[P"DV,`!1Z#.P4`8XP+'%Z%%W,```,SHZ$"
MDY,4%0"%!":%%W,```,S`Z$"DP/'"P`JC*:,JHH=SQ=[```#.ZNYEQ8``).&
M)M<FAHE%8H4"FX4+YH9C\YP`IH:SAY9`B0=C\]<`MH<#QPL`^125C+Z4"0QY
M]Y,$A.T7%0``$P4ERB&@B&2A!`7%RH7OX)_8*HMMV8-'!0#UU^_P#X1]T5J%
M%W,```,S(ZL"DZJ$'>6#-`3L$PL$[)=[``"#N\NID<R#QP0`8XT'$":%[_#O
M@&,2!1`A"X,T"P#E]!=S```#,P.5)H4"DRJ+%W,```,S(Y16A0*3.PNK`!L+
M2T`7<P```S-CDUJ%`I.7%P``DX<GM=:()H=:AA<8```3"$C*EQ8``).&1LJ)
M11=S```#,Z.JJHL"DUZ%DP4`'!=S```#,X.@`I/]5V,;]0`7<P```S,#D`*3
M'$%%1V.;YQ"3!83P7H47<P```S,#H@*3JH0)Z8,GA/&11A/7QT`]BV,#UPR7
M=P``@[<GGH,V"@"(8UZ'%Q8``!,&)LB)11=S```#,P.B`I.!2@,WA/B#MPD`
MN8\!1V.<!SCV<%9T%FOR>U)\LGRV=!9Y]FE6:E:%MFH988*`)H6"FZJ$"?$A
M"^6]@43MO0,UA.J3!(3JEWH``(.ZBI89R<J%[^#?P0'%@T<%`(WGB&2A!'WU
M-447<P```S.#@@*3M4PJC*J*YH27&P``DXOKMA,',`4!O8*:JHMQV1=C```#
M,Z-_`I.3%!4`A03%LX,J!/(78P```S/#?0*3XYBJ\H,GA/%J\F[N$P<`')/W
M]Q]CC^<$$GWR;1&_%W<```,W1X^#-@H`"&,7%@``$P9FM5Z'B447<P```S,C
MDP*3";_O\#^Z`S>$^(.W"0"YCP%'8YH'*E9T]G`6>?9I5FHFA;9T%W,```,S
MPXD980*#%V,```,S`W=:A0*3RH6JBA<5```3!86Y[^"?LZJ%8P8%%`,U"@#O
M\&^;JHQC#P429H478P```S-C<P*3&PP%`(U'8]N'`Q,%S/]FE9<5``"3A46V
M%V,```,SPWT"DPGMD4=C"OP<FP>\_^:7`\<'`),'\`)C`?<<RH47%0``$P7E
ME^_@_ZQMQ8-'!0#UPY.'!_WYSY<5``"3A<6?%Q4``!,%Q97OX#^V%V,```,S
M@W<"DRJ(7H=:AE:%EQ@``).(Z+Z7%P``DX=GC9<6``"3AF:QB447<P```S/#
M@P*3%VH```,Z*GI]69=L``"#O`QJ14R7;0``@[WM<A<=```3#>VZ':B"G!Q!
M8YZ'`X*=A20JB&K@IHA>AUJ&5H67%P``DX?'AY<6``"3AL:LB4478P```S,C
M?@*3DP4`'%:%`IKC`B7]%V,```,SPW)>A0*3UH47%0``$P7E@>_@WZI6A>_P
M7Z$2??)MM;N#/`H`R;5FA8%%%V,```,S0WT"DRJ)$=GO\/^&*HKC1@7PDP5E
M_P%&%V,```,SXVQ*A0*3DP4$ZDJ%&4878P```S.C<P*3F4=C'O4*`R<$ZK='
M04.3AP<PXQGW[`-71.J19Y.'AU3C$O?LDP7J_`%&%V,```,S8VA*A0*32H63
M!83P$P:``A=C```#,P-O`I.3!X`"8QCU!B,(!/(3"(3P6H:7&```DXC(J9<'
M``"3AT=X7H>7%@``DX8FFXE%5H478P```S-C;@*3`;\#.HH`XP\*XE*%%V,`
M``,S@U("DXU'X_:GXG$54I67%0``DX5%EA=C```#,T-=`I/C&07@THPQM0,W
MA/B#MPD`N8\!1XGK%Q4``!,%193O\._P:O)N[A=C```#,\-J`I/6YMKB7OYB
M^F;V:O)N[A=C```#,T-I`I-=<2;\EV0``(.T9%:BX%+PAN1*^%;L@`BSAT0`
M`[D'`!=J```#.BI&DP5$^X-'"0L7!0``$P6E;P,W"@`C/.3Z`4>C"/D*[^#?
MA1G%`T<%`),'$`-C!_<6[_`/I;.'1``#N0<`2H6-1@5&EQ4``).%!8\78P``
M`S,#20*3*HD=Q1QI,X=$``ACC&,7%@``$P:&C1=C```#,X-'`I.#-PD!C&/<
M11.7EP)C20<2LX=$`(ACC48%1I<5``"3A46+%V,```,S0T0"DRJ)EVH``).*
M"F,5Q9,%1/L7!0``$P6E7^_@+_R#-PD!*H:,8V7!LX=$`(AC%V,```,S8T$"
MD[.'1``#N0<`!4:7%0``DX6EAA=C```#,R,Y2H4"DZJ%%Q8``!,&QH9*A1=C
M```#,R,^`I.2E)Q@DP5$^Q<%```3!858HX@'"N_@S_4CN*H`JH0AP9,%`!PF
MA1=C```#,Z-'`I,;!Q4`A4=C]><`DP?U_K7O`S>$^X,W"@"YCP%'R>^F8`9D
MXG1">0)ZXFIA88*``[6*`._P_X(CN*H`JH15^0,WA/N#-PH`N8\!1T[TM><7
M!0``$P4E?A=C```#,Z-&`I.#1Q4`XX\'Z'FU@[>*`"Z%C&/OX*_<';>SAT0`
MB&,78P```S.C00*3R;4#-X3[@S<*`+F/`4=.](7C@[>*`*J&)H:,8Q<%```3
M!45[%V,```,S0T$"DT[T%V,```,S`T<"DSEQ(O@F]$KP@``&_)=D``"#M"0E
M*HF3!43]%P4``!,%Q5"<8",\]/R!1^_@3^4%P8-'!0`3AP?]$S<7`).W%P#9
MCY'G8P4)`(-'"0")[P,WA/V<8+F/`4>]Y^)P0G2B=`)Y(6&"@$J%[^#_[9,%
M\`(78P```S-C-R,TI/P"DX,WA/P9P9,'%0`78P```S-C,SZ%`I,11I<%``"3
MA059%V,```,S@QT"DUWQ`S>$_9Q@N8\!1XGK0G3B<*)T2H4">2%A;^#?CQ=C
M```#,P,Z`I-!$2+@!N0`"!,%``@78P```S,C'`*3ET4``(.UA4DW=UU,MW?I
M808'-R9%9[=6,A"&!Q,'Y\\3!A8PDX9F1Y.'!Q\8Z2,T!0(C.`4"(RP%!A#A
M#.44[1SQHF`"9$$!@H`=<:+HRN!._%;T6O"&[*;D4OB`$!AUDQ<V`+*)NI<0
M>8(7@9,SM^<`*HF3UMD!*%TZEC.'Q@`C-/D"(SCI`BZ+DPJ)`UGEDP?P`V/0
M-P<;VFD`8N@;#/K_9N0:"I,<#`)>[)/<#`):FMJ$EVL``(.[2QJFA1,&``16
MA8*;DX0$!$J%[]`OF..61/\[#(!!&QQL`)N)"?R3EVP`NPD\`>)K0FRB;!,+
M"P0^F]J%5H5.AA=C```#,^,5`I,C+#D'YF!&9*9D!FGB>4)ZHGH">R5A@H"3
M!P`$B9^^A&/.^0(;B@0`DPJ)`U*&5I7:A1=C```#,R,2`I.#)XD'$P<`!*6?
M(RSY!N.=Y_I*A;N)F4!2F^_03X\-O\Z$V;==<:+@)OQ*^%;L6NB&Y$[T4O"`
M"(.YA0*#UH4"`]<%`Y/7.0"NA)/W]P.3A84#,X;U`!,(`/B3VH8`$UN'``.Z
M!`.%)R,`!@&;EH8`&Q>'`!,&@`,JB;::,X7U`#J;8U[V$!,'``2[!_=`$4=C
M^^<2"4=C_^<08YH'$":%(SRT^N_0KX8#-83[$P:``X%%%V,```,SXQ,"DY/7
MB0$35XH!D]D)`1-:"@$CF60'(YM4!Z.(1`<CBO0&HXHT!R:%(XCD!N_0CX*<
M8":%%V,```,SHPCA@R,`^0"<8,&#HP#Y`)Q@H8,C`?D`G&"C`?D`G&3A@R,"
M^0"<9,&#HP+Y`)QDH8,C`_D`G&2C`_D`G&CA@R,$^0"<:,&#HP3Y`)QHH8,C
M!?D`G&BC!?D`G&SA@R,&^0"<;,&#HP;Y`)QLH8,C!_D`G&RC!_D`G'#A@R,(
M^0"<<,&#HPCY`)QPH8,C"?D`G'"C"?D`!F2F8.)T0GFB>0)ZXFI":V%A`H,=
MGH%%%V,```,SXP0"DP&_(P`%`/6U(P`%`*,`!0#Y%SJ5XX\'[.VW(P`%`*,`
M!0`C`04`HP$%`/$7.I4)1^/OY^K1OT$1(N`&Y``(`F2B8)=F``"#MJ8&00%O
MP!_!``````$``@``````+W!R;V,O<V5L9B]E>&4``"5S+R5S````1'EN84QO
M861E<CHZ8F]O=%]$>6YA3&]A9&5R``````!);G1E<FYA;',Z.E!!4CHZ0D]/
M5`````!E>&4``````"]P<F]C+R5I+R5S```````N`````````%!!4E]414U0
M``````````!005)?4%)/1TY!344`````.@`````````O`````````"5S)7,E
M<P``4$523#5,24(``````````%!%4DQ,24(`4$523#5/4%0``````````%!%
M4DQ)3P``4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$``````!005)?0TQ%
M04X`````````4$%27T1%0E5'`````````%!!4E]#04-(10````````!005)?
M1TQ/0D%,7T1%0E5'``````````!005)?1TQ/0D%,7U1-4$1)4@````````!0
M05)?5$U01$E2````````4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%
M04X``````````#$`````````;'-E96L@9F%I;&5D`````')E860@9F%I;&5D
M``````!,1%],24)205)97U!!5$@`4UE35$5-```O=&UP`````%5315(`````
M55-%4DY!344``````````"4P,G@`````<&%R+0`````E<R5S)7,E<P``````
M````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A
M:6QE9"`H97)R;F\])6DI"@```````"5S.B!P<FEV871E('-U8F1I<F5C=&]R
M>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO
M=7(@;W!E<F%T:6]N*0H```````!0051(`````'!A<FP`````+G!A<@````!S
M:&]R="!R96%D````````)7,E<V-A8VAE+25S)7,``"5S)7-T96UP+25U)7,`
M```E<R5S=&5M<"TE=2TE=25S```````````8`````````'!E<FP`````,```
M``````!005(Z.E!A8VME<CHZ5D524TE/3@`````Q+C`V-````%5N86)L92!T
M;R!C<F5A=&4@8V%C:&4@9&ER96-T;W)Y```````````E<SH@8W)E871I;VX@
M;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N
M;STE:2D*```````````M90```````"TM````````979A;"`G97AE8R`O=7-R
M+V)I;B]P97)L("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@;F]T(')U;FYI
M;F<@=6YD97(@<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L.PIM>2`H)%!!
M4E]-04=)0RP@)$9)3$5?;V9F<V5T7W-I>F4L("1C86-H95]N86UE7W-I>F4I
M.PI"14=)3B!["B1005)?34%'24,@/2`B7&Y005(N<&U<;B(["B1&24Q%7V]F
M9G-E=%]S:7IE(#T@-#L@("`C('!A8VLH(DXB*0HD8V%C:&5?;F%M95]S:7IE
M(#T@-#`["B102UI)4%]-04=)0R`](")02UPP,#-<,#`T(CL*?0IS=6(@9FEN
M9%]P87)?;6%G:6,*>PIM>2`H)&9H*2`]($!?.PIM>2`D8VAU;FM?<VEZ92`]
M(#8T("H@,3`R-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D9F@["FUY("1P
M;W,@/2`H)'-I>F4M,2D@+2`H)'-I>F4M,2D@)2`D8VAU;FM?<VEZ93L@("`@
M("`C($Y/5$4Z("1P;W,@:7,@82!M=6QT:7!L92!O9B`D8VAU;FM?<VEZ90IW
M:&EL92`H)'!O<R`^/2`P*2!["G-E96L@)&9H+"`D<&]S+"`P.PIR96%D("1F
M:"P@)&)U9BP@)&-H=6YK7W-I>F4@*R!L96YG=&@H)%!!4E]-04=)0RD["FEF
M("@H;7D@)&D@/2!R:6YD97@H)&)U9BP@)%!!4E]-04=)0RDI(#X](#`I('L*
M<F5T=7)N("1P;W,@*R`D:3L*?0HD<&]S("T]("1C:'5N:U]S:7IE.PI]"G)E
M='5R;B`M,3L*?0IM>2`H)'!A<E]T96UP+"`D<')O9VYA;64L($!T;7!F:6QE
M+"`E36]D=6QE0V%C:&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2![
M"G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*
M<F5Q=6ER92!&:6QE.CI3<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA
M;64Z.F1I<FYA;64H)'!A<E]T96UP*3L*;W5T<RAQ<5M296UO=FEN9R!F:6QE
M<R!I;B`B)'!A<E]T96UP(ETI.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H<W5B
M('L@*"`M9"`I(#\@<FUD:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*<FUD
M:7(@)'!A<E]T96UP.PII9B`H+60@)'!A<E]T96UP("8F("1>3R!N92`G35-7
M:6XS,B<I('L*;7D@)'1M<"`](&YE=R!&:6QE.CI496UP*`I414U03$%412`]
M/B`G=&UP6%A86%@G+`I$25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H
M)'1O<&1I<BDL"E-51D9)6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI.PIM
M>2`D9FEL96YA;64@/2`D=&UP+3YF:6QE;F%M93L*<')I;G0@)'1M<"`\/"(N
M+BXB.PIX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D;PIR;2`M<F8@)R1P
M87)?=&5M<"<*:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=.R!T:&5N"F)R96%K
M"F9I"G-L965P(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE"G)M("<D9FEL96YA
M;64G"BXN+@IC;&]S92`D=&UP.PIC:&UO9"`P-S`P+"`D9FEL96YA;64["FUY
M("1C;60@/2`B)&9I;&5N86UE(#XO9&5V+VYU;&P@,CXF,2`F(CL*<WES=&5M
M*"1C;60I.PIO=71S*'%Q6U-P87=N960@8F%C:V=R;W5N9"!P<F]C97-S('1O
M('!E<F9O<FT@8VQE86YU<#H@)&9I;&5N86UE72D["GT*?2!]"D)%1TE.('L*
M26YT97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA;',Z
M.E!!4CHZ0D]/5#L*979A;"!["E]P87)?:6YI=%]E;G8H*3L*;7D@)'%U:65T
M(#T@(21%3E9[4$%27T1%0E5'?3L*;7D@)7-Y<R`]("@*<&%T:%]S97`@("`@
M/3X@*"1>3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"E]E>&4@("`@("`@
M(#T^("@D7D\@/7X@+UXH/SI-4U=I;GQ/4S)\8WEG=VEN*2\@/R`G+F5X92<@
M.B`G)RDL"E]D96QI;2`@("`@(#T^("@D7D\@/7X@+UY-4U=I;GQ/4S(O(#\@
M)UQ<)R`Z("<O)RDL"BD["E]S971?<')O9VYA;64H*3L*;W5T<RAQ<5M<)'!R
M;V=N86UE(#T@(B1P<F]G;F%M92)=*3L*7W-E=%]P87)?=&5M<"@I.PIO=71S
M*'%Q6UPD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0?2)=*3L*;7D@
M*"1S=&%R=%]P;W,L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI.PI-04=)0SH@
M>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![?3L*=6YL97-S("AO<&5N
M(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE*2!["F]U=',H<7%;0V%N)W0@<F5A
M9"!F<F]M(&9I;&4@(B1P<F]G;F%M92)=*3L@(",@9&]N)W0@=7-E("0A(&AE
M<F4@87,@:70@<F5Q=6ER97,@17)R;F\N<&T*;&%S="!-04=)0SL*?0IM>2`D
M;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H*E]&2"D["FEF("@D;6%G:6-?
M<&]S(#P@,"D@>PIO=71S*'%Q6T-A;B=T(&9I;F0@;6%G:6,@<W1R:6YG("(D
M4$%27TU!1TE#(B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,[
M"GT*;W5T<R@B1F]U;F0@4$%2(&UA9VEC(&%T('!O<VET:6]N("1M86=I8U]P
M;W,B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ
M92`M(&QE;F=T:"@B7#!#04-(12(I+"`P.PIR96%D(%]&2"P@)&)U9BP@;&5N
M9W1H*")<,$-!0TA%(BD["FEF("@D8G5F(&YE(")<,$-!0TA%(B`F)B`D8G5F
M(&YE(")<,$-,14%.(BD@>PIO=71S*").;R!C86-H92!M87)K97(@9F]U;F0B
M*3L*;&%S="!-04=)0SL*?0IO=71S*'%Q6T-A8VAE(&UA<FME<B`B)&)U9B(@
M9F]U;F1=*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?
M<VEZ92P@,#L*<F5A9"!?1D@L("1B=68L("1&24Q%7V]F9G-E=%]S:7IE.PIM
M>2`D;V9F<V5T(#T@=6YP86-K*").(BP@)&)U9BD["F]U=',H(D]F9G-E="!F
M<F]M('-T87)T(&]F($9)3$5S(&ES("1O9F9S970B*3L*<V5E:R!?1D@L("1M
M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92`M("1O9F9S970L(#`["B1S
M=&%R=%]O9E]&24Q%7W-E8W1I;VX@/2!T96QL(%]&2#L*;7D@)7)E<75I<F5?
M;&ES=#L*<F5A9"!?1D@L("1B=68L(#0[("`@("`@("`@("`@("`@("`@("`@
M("`@("`@(R!R96%D('1H92!F:7)S="`B1DE,12(*=VAI;&4@*"1B=68@97$@
M(D9)3$4B*2!["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@
M=6YP86-K*").(BP@)&)U9BD["FUY("1F=6QL;F%M92`]("1B=68["F]U=',H
M<7%;56YP86-K:6YG($9)3$4@(B1F=6QL;F%M92(N+BY=*3L*;7D@)&-R8R`]
M("@@)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I(#\@)#$@.B!U
M;F1E9CL*;7D@*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ
M+RD_*"XJ*2A<+BXJ*7PI.PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L
M("1B=68L('5N<&%C:R@B3B(L("1B=68I.PII9B`H9&5F:6YE9"@D97AT*2!A
M;F0@)&5X="`A?B`O7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*;7D@)&9I;&5N
M86UE(#T@7W-A=F5?87,H(B1C<F,D97AT(BP@)&)U9BP@,#<U-2D["B1005(Z
M.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M93L*)%!!
M4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN86UE.PI]
M"F5L<V4@>PHD36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B
M=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*?3L*)')E<75I
M<F5?;&ES='LD9G5L;&YA;65]*RL["GT*<F5A9"!?1D@L("1B=68L(#0["GT*
M;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U;&4I(#T@0%\[
M"G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*;7D@)&EN9F\@
M/2`D36]D=6QE0V%C:&5[)&UO9'5L97T["G)E='5R;B!U;FQE<W,@)&EN9F\[
M"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5].PHD24Y#>R1M;V1U;&5]
M(#T@(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!
M3GT@86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/
M.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@(D-A;B=T(&-R96%T92!T96UP
M(&9I;&4Z("0A(CL*)&9H+3YB:6YM;V1E*"D["B1F:"T^<')I;G0H)&EN9F\M
M/GMB=69]*3L*)&9H+3YS965K*#`L(#`I.PIR971U<FX@)&9H.PI]"F5L<V4@
M>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&EN9F\M/GMC<F-]+G!M(BP@
M)&EN9F\M/GMB=69]*3L*)&EN9F\M/GMF:6QE?2`]("1F:6QE;F%M93L*;W!E
M;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE;F%M92!O<B!D:64@<7%;0V%N)W0@
M<F5A9"`B)&9I;&5N86UE(CH@)"%=.PIR971U<FX@)&9H.PI]"F1I92`B0F]O
M='-T<F%P<&EN9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D=6QE("1M;V1U;&4A
M(CL*?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(
M96%V>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ
M2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER
M92!)3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@
M)7)E<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@
M;W(@)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@
M+UPN<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@
M)&9I;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<
M+G!L)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]
M"FEF("@D8G5F(&YE("102UI)4%]-04=)0RD@>PIO=71S*'%Q6TYO('II<"!F
M;W5N9"!A9G1E<B!&24Q%('-E8W1I;VX@:6X@9FEL92`B)'!R;V=N86UE(ETI
M.PIL87-T($U!1TE#(#L*?0HD<W1A<G1?<&]S(#T@*'1E;&P@7T9(*2`M(#0[
M("`@("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@0'!A<E]A<F=S
M.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M92D["F1E
M;&5T92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2155312!M
M87D@8F4@82!S96-U<FET>2!P<F]B;&5M*0II9B`H(21S=&%R=%]P;W,@;W(@
M*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@
M)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P
M87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?
M<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2![
M"G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%2
M1U8["GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?
M=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/
M0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP
M=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V
M86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P
M87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#([
M"GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L
M<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D
M,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2![
M"F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L
M;V<@9FEL92`B)#(B.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C
M:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D
M9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET
M7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD
M(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F
M;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC
M.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E
M($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R
M(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2![
M"F]P96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F
M:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL
M92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R
M;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/
M2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P
M87(B73L*?0IM>2`E;65T85]P87(@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D
M;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^
M(',O+BI><&%R.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@
M/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG.PI]"GT["FEF("AD969I;F5D("1P
M87(I('L*;W!E;B!M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A
M;B=T(')E860@<&%R(&9I;&4@(B1P87(B.B`D(5T["FUY("1B=68["G)E860@
M)'!H+"`D8G5F+"`T.PID:64@<7%;(B1P87(B(&ES(&YO="!A('!A<B!F:6QE
M72!U;FQE<W,@)&)U9B!E<2`D4$M:25!?34%'24,["F-L;W-E("1P:#L*?0I#
M<F5A=&50871H*"1O=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=R@*)&]U
M="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE.CI/7U)$5U(H*2!\
M($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A;B=T
M(&-R96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*<V5E
M:R!?1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD969I;F5D("1S=&%R=%]O
M9E]&24Q%7W-E8W1I;VXI('L*<F5A9"!?1D@L("1L;V%D97(L("1S=&%R=%]O
M9E]&24Q%7W-E8W1I;VX["GT@96QS92!["FQO8V%L("0O(#T@=6YD968["B1L
M;V%D97(@/2`\7T9(/CL*?0II9B`H(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@
M)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I
M;'1E<CHZ4&]D4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH
M7"1L;V%D97(L("0P*3L*?0HD9F@M/G!R:6YT*"1L;V%D97(I"F]R(&1I92!Q
M<5M%<G)O<B!W<FET:6YG(&QO861E<B!T;R`B)&]U="(Z("0A73L*:68@*"1B
M=6YD;&4I('L*<F5Q=6ER92!005(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI
M=%]D>6YA;&]A9&5R*"D["FEN:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I
M.PIM>2!`:6YC(#T@9W)E<"![("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L
M92!N92`G<VET92<I(&]R"B@D7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI
M8F5X<'T@86YD"B1?(&YE("1#;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D[
M"GT@0$E.0SL*<WPO*WPO?&<L('-\+R1\?"!F;W)E86-H($!I;F,["FUY("5F
M:6QE<SL*)&9I;&5S>R1??2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D
M7V]B:F5C=',L('9A;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z
M.D-O;F9I9WML:6)?97AT?3L@("`@("`@("`C(%A86"!L:6)?97AT('9S(&1L
M97AT(#\*;7D@)7=R:71T96X["F9O<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES
M("5F:6QE<RD@>PIM>2`H)&9I;&4L("1N86UE*3L*:68@*&1E9FEN960H;7D@
M)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE
M+"`D;F%M92D@/2`H)&ME>2P@)&9C*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D
M9&ER("A`:6YC*2!["FEF("@D:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI
M('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D
M:V5Y(#U^(&U\7B]L;V%D97(O6UXO72LO*"XJ*21\*2!["FEF("AM>2`D<F5F
M(#T@)$UO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F
M+"`D,2D["FQA<W0["GT*:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N
M86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N
M;&5S<R!D969I;F5D("1N86UE.PIN97AT(&EF("1W<FET=&5N>R1N86UE?2LK
M.PIN97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?
M97AT7$4D+VD["F]U=',H<W!R:6YT9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN
M+ETL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D
M8V]N=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^
M>V)U9GT["GT*96QS92!["FQO8V%L("0O(#T@=6YD968["F]P96X@;7D@)'1H
M+"`G/#IR87<G+"`D9FEL92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B
M.B`D(5T["B1C;VYT96YT(#T@/"1T:#X["F-L;W-E("1T:#L*4$%2.CI&:6QT
M97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O;G1E;G0L("(\96UB961D960^+R1N
M86UE(BD*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN
M*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN
M97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R
M:6YT*")&24Q%(BP*<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R
M:6YT9B@B)3`X>"\E<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C
M;VYT96YT*2P@)&YA;64I+`IP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL
M"B1C;VYT96YT*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!E;6)E9&1E9"!&
M24Q%('1O("(D;W5T(CH@)"%=.PIO=71S*'%Q6U=R:71T96X@87,@(B1N86UE
M(ETI.PI]"GT*:68@*"1Z:7`I('L*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H
M)&9H*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+"F]R(&1I92!Q<5M%<G)O<B!W
M<FET:6YG('II<"!P87)T(&]F("(D;W5T(ET["GT*:68@*"1M971A7W!A<GMC
M;&5A;GTI('L*)&9H+3YP<FEN="@B7#!#3$5!3B(I.PI]"F5L<V4@>PII9B`H
M(61E9FEN960@)&-A8VAE7VYA;64I('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!
M+3YN97<H,2D["G-E96L@)&9H+"`P+"`P.PHD8W1X+3YA9&1F:6QE*"1F:"D[
M"G-E96L@)&9H+"`P+"`R.PHD8V%C:&5?;F%M92`]("1C='@M/FAE>&1I9V5S
M=#L*?0HD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@)&-A
M8VAE7VYA;65?<VEZ93L*)&-A8VAE7VYA;64@+CT@(EPP(B!X("@D8V%C:&5?
M;F%M95]S:7IE("T@;&5N9W1H("1C86-H95]N86UE*3L*)&9H+3YP<FEN="@D
M8V%C:&5?;F%M92P*(EPP0T%#2$4B*3L*?0IM>2`D;V9F<V5T(#T@)&9H+3YT
M96QL("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D
M;V9F<V5T*2P*)%!!4E]-04=)0RD["B1F:"T^8VQO<V4@;W(@9&EE('%Q6T5R
M<F]R('=R:71I;F<@=')A:6QE<B!O9B`B)&]U="(Z("0A73L*8VAM;V0@,#<U
M-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T
M7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V
M>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*
M<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE
M=SL@("`@("`@("`@("`@("`@("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP
M(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO
M*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!
M<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@
M07)C:&EV93HZ6FEP+3YN97<["B@D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H
M)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@
M9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=
M.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@V-"`J(#$P,C0I.PIP=7-H
M($!005(Z.DQI8D-A8VAE+"`D>FEP.PHD4$%2.CI,:6)#86-H97LD<')O9VYA
M;65](#T@)'II<#L*;W5T<R@B17AT<F%C=&EN9R!Z:7`N+BXB*3L*:68@*&1E
M9FEN960@)$5.5GM005)?5$5-4'TI('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H
M:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@
M*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["FUY("1M96UB97)?
M;F%M92`]("1M96UB97(M/F9I;&5.86UE.PIN97AT('5N;&5S<R`D;65M8F5R
M7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD0V]N9FEG.CI#;VYF:6=[87)C
M:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?;F%M92`]("0Q
M.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL92@D14Y6>U!!
M4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S=%]N86UE("8F
M("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I*2!["F]U=',H
M<7%;4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@86QR96%D>2!E
M>&ES=',@870@(B1D97-T7VYA;64B72D["GT@96QS92!["F]U=',H<7%;17AT
M<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M92(N+BY=*3L*
M*"1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE*2`]/2!!
M<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!E>'1R86-T
M:6YG('II<"!M96UB97(@=&\@(B1D97-T7VYA;64B73L*8VAM;V0H,#4U-2P@
M)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*?0IU;FQE<W,@
M*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\("(N(B!U;FQE
M<W,@0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;
M("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L(%T*)#`@6R`M
M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!4E]04D]'3D%-
M17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*?0IS=6(@0W)E
M871E4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&:6QE.CI"87-E
M;F%M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@1FEL93HZ0F%S
M96YA;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR97%U:7)E($9I
M;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@=6YL97-S*"UE
M("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*<W5B(')E<75I
M<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y;F%,;V%D97([
M"G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E<75I<F4@=V%R
M;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR97%U:7)E($-A
M<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X<&]R=&5R.CI(
M96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT;#L*<F5Q=6ER
M92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E<75I<F4@6%-,
M;V%D97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(86YD;&4["G)E
M<75I<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL:6(["G)E<75I
M<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PIR97%U:7)E
M(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E(%!!4CHZ1&ES=#L*
M<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I<F4@4$%2.CI&
M:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U=&5S.PIE=F%L
M('L@<F5Q=6ER92!#=V0@?3L*979A;"![(')E<75I<F4@5VEN,S(@?3L*979A
M;"![(')E<75I<F4@4V-A;&%R.CI5=&EL('T["F5V86P@>R!R97%U:7)E($%R
M8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER92!4:64Z.DAA
M<V@Z.DYA;65D0V%P='5R92!].PIE=F%L('L@<F5Q=6ER92!097)L24\[(')E
M<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*979A;"![(')E<75I<F4@=71F."!]
M.PI]"G-U8B!?<V5T7W!A<E]T96UP('L*:68@*&1E9FEN960@)$5.5GM005)?
M5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+RD@>PHD<&%R7W1E
M;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@*`HH;6%P("1%
M3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!435!$25(@5$5-4$1)4B!414U0(%1-
M4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@+B`I"BD@>PIN97AT('5N;&5S<R!D
M969I;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A=&@["FUY("1U
M<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`](&=E='!W=6ED
M*"0^*2!I9B!D969I;F5D("0^.WT["FEF("@@9&5F:6YE9"@F5VEN,S(Z.DQO
M9VEN3F%M92D@*2!["B1U<V5R;F%M92`]("97:6XS,CHZ3&]G:6Y.86UE.PI]
M"F5L<VEF("AD969I;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@/2`D<'=U:60[
M"GT*96QS92!["B1U<V5R;F%M92`]("1%3E9[55-%4DY!345]('Q\("1%3E9[
M55-%4GT@?'P@)U-94U1%32<["GT*;7D@)'-T;7!D:7(@/2`B)'!A=&@D<WES
M>U]D96QI;7UP87(M(BYU;G!A8VLH(D@J(BP@)'5S97)N86UE*3L*;6MD:7(@
M)'-T;7!D:7(L(#`W-34["FUY("1C86-H95]D:7(["FEF("@D14Y6>U!!4E]#
M3$5!3GTI('L*)&-A8VAE7V1I<B`](")T96UP+20D(CL*?0IE;'-E('L*;W!E
M;B!M>2`D9F@L("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@<7%;0V%N)W0@
M<F5A9"`B)'!R;V=N86UE(CH@)"%=.PII9B`H*&UY("1M86=I8U]P;W,@/2!F
M:6YD7W!A<E]M86=I8R@D9F@I*2`^/2`P*2!["G-E96L@)&9H+"`D;6%G:6-?
M<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*2P@
M,#L*;7D@)&)U9CL*<F5A9"`D9F@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I
M.PII9B`H)&)U9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L("1M86=I8U]P
M;W,*+2`D1DE,15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#04-(12(I"BT@
M)&-A8VAE7VYA;65?<VEZ92P@,#L*<F5A9"`D9F@L("1B=68L("1C86-H95]N
M86UE7W-I>F4["B1B=68@/7X@<R]<,"\O9SL*)&-A8VAE7V1I<B`](")C86-H
M92TD8G5F(CL*?0I]"F-L;W-E("1F:#L*?0II9B`H(21C86-H95]D:7(I('L*
M)&-A8VAE7V1I<B`](")T96UP+20D(CL*)$5.5GM005)?0TQ%04Y](#T@,3L*
M?0HD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U])&-A8VAE7V1I<B(["FUK9&ER
M("1S=&UP9&ER+"`P-S4U.PHD14Y6>U!!4E]414U0?2`]("1S=&UP9&ER.PIL
M87-T.PI]"B1P87)?=&5M<"`]("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D
M14Y6>U!!4E]414U0?2`]?B`O*"XK*2\["GT*<W5B(%]S879E7V%S('L*;7D@
M*"1N86UE+"`D8V]N=&5N=',L("1M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@
M/2`B)'!A<E]T96UP+R1N86UE(CL*=6YL97-S("@M92`D9G5L;&YA;64I('L*
M;7D@)'1E;7!N86UE(#T@(B1F=6QL;F%M92XD)"(["FUY("1F:#L**&]P96X@
M)&9H+"`G/CIR87<G+"`D=&5M<&YA;64I"B8F("AP<FEN="`D9F@@)&-O;G1E
M;G1S*0HF)B`H8VQO<V4@)&9H*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R`B
M)'1E;7!N86UE(CH@)"%=.PIC:&UO9"`D;6]D92P@)'1E;7!N86UE(&EF(&1E
M9FEN960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L("1F=6QL;F%M92D@;W(@
M=6YL:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F=6QL;F%M93L*?0IS=6(@
M7W-E=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%3E9[4$%27U!23T=.04U%
M?2!A;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI+RD@>PHD<')O9VYA
M;64@/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII9B`H)$5.5GM005)?5$5-
M4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI(#X](#`I
M('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G
M;F%M92P@)'-Y<WM?9&5L:6U]*2`K(#$I.PI]"FEF("@A)$5.5GM005)?4%)/
M1TY!345](&]R(&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`^/2`P
M*2!["FEF("AO<&5N(&UY("1F:"P@)SPG+"`D<')O9VYA;64I('L*<F5T=7)N
M(&EF("US("1F:#L*?0II9B`H+7,@(B1P<F]G;F%M921S>7-[7V5X97TB*2![
M"B1P<F]G;F%M92`N/2`D<WES>U]E>&5].PIR971U<FX["GT*?0IF;W)E86-H
M(&UY("1D:7(@*'-P;&ET("]<421S>7-[<&%T:%]S97!]7$4O+"`D14Y6>U!!
M5$A]*2!["FYE>'0@:68@97AI<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER
M(&5Q("1%3E9[4$%27U1%35!].PHD9&ER(#U^(',O7%$D<WES>U]D96QI;7U<
M120O+SL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M
M921S>7-[7V5X97TB*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD
M<')O9VYA;64D<WES>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[
M7V1E;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E
M;&EM?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P(#T@
M)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E>"@D
M<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M92`]("(N
M)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@*&1E9FEN960@
M)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E9FEN960@)E=I
M;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W9&`I.PIC:&]M
M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D<WES>U]D96QI
M;7U<12DO)'!W9"1S>7-[7V1E;&EM?2\["B1%3E9[4$%27U!23T=.04U%?2`]
M("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N=B!["FEF("@@)$5.5GM0
M05)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD
M14Y6>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O<B`H<7<H(%-005=.140@
M5$5-4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-12`I("D@>PID96QE=&4@
M)$5.5GLB4$%27R1?(GT["GT*9F]R("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@
M1$5"54<@+RD@>PHD14Y6>R)005)?)%\B?2`]("1%3E9[(E!!4E]'3$]"04Q?
M)%\B?2!I9B!E>&ES=',@)$5.5GLB4$%27T=,3T)!3%\D7R)].PI]"FEF("@D
M14Y6>U!!4E]414U0?2D@>PID96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"GT*
M<W5B(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO9V9H*2!["G!R
M:6YT("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`7UQN(CL*?0I]
M"G-U8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R
M97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H
M;&EB97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI8F5X<`IV96YD
M;W)A<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q
M=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R.PI005(M
M/FEM<&]R="A`<&%R7V%R9W,I.PID:64@<7%;<&%R+G!L.B!#86XG="!O<&5N
M('!E<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER
M96-T;W)Y7&Y="G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P<F]G;F%M93L*
M0T]213HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO
M*3L*9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@:68@)$`["GT*
M0T]213HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH
M*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?
M7T5.1%]?"@!435!$25(``%1%35!$25(`5$5-4`````!435```````(FKS>\`
M````<&5R;'AS:2YC``````````!#04-(10```1L#.^P````<````L(K__S`'
M``"DC/__!`$``):-__\8`0``W)___W`!``#@H/__K`$``):A__\<`@``IJ/_
M_UP"```@I/__E`(``*:E__\``P``$J;__RP#``!.IO__5`,``%RG__^0`P``
M;*?__[@#``!TJ?__#`0``*BI__\X!```YJK__X`$``#`KO__N`0``"*O___H
M!```@*___P@%``#<L/__7`4```2Q__^$!0```++__]P%```^N/__G`8``&*Z
M___H!@``,KO__VP'``"2N___E`<``*R\___H!P``:K[__S`(````````$```
M```````#>E(``7P!`1L,`@`0````&````)B+__\B``````<!`%0````L````
M=HS__T82````1`[P!42(!$(,"`!LEQ*!`HD&D@B3"I0,E0Z6$)@4F1::&`/4
M$0K!1,@,`O`%1,E$TD331-1$U4361-=$V$391-I$#@!""P`X````A````&2>
M__\$`0```$(.4$R(!($"B0:2"),*E`Q"#`@``HH*#`)00L%"R$+)0M)"TT+4
M0@X`0@ML````P````"R?__^V`````$(.4$B(!($"B0:6$$(,"`!6DPI:D@B4
M#)4.EQ*8%'+20M-"U$+50M="V$+!0L@,`E!"UD3)0@X`0@T(@0*(!(D&DPJ6
M$%:2")0,E0Z7$I@44M+3U-77V$23"D+3````/````#`!``!RG___$`(```!"
M#G!.B`2)!I((DPJ!`I0,E0Y"#`@``P@!"L%"R`P"<$+)0M)"TT+40M5"#@!"
M"S0```!P`0``0J'__WH`````0@ZP`4:(!(D&D@A"#`@`2H$"`E(*P4+(#`*P
M`4+)0M)"#@!""P``:````*@!``"$H?__A@$```!"#N`!1H@$DPJ!`D(,"`!:
MB09&R5+!0L@,`N`!0M-"#@!"#0B!`H@$B0:3"DJ2"$J4#`)4E0X"D-5>"LE"
MTD+40@M&E0Y(R=+4U4B)!I((E`R5#DK4U4+)0M(`*````!0"``">HO__;```
M``!"#B!"B09,B`2!`D(,"`!LR`P"($3!0LEF#@`D````0`(``-ZB__\\````
M`$H.$$2(!($"0@P(`$3(#`(00L%D#@``.````&@"``#RHO__#@$```!"#F!,
MB`2!`HD&D@B3"I0,0@P(``+B"@P"8$+!0LA"R4+20M-"U$0.`$(+)````*0"
M``#$H___$`````!"#A!$B`2!`D(,"`!"R`P"$$+!0@X``%````#,`@``K*/_
M_P@"````0@[P`5:(!($"B0:2"),*E`R5#I80EQ*8%)D60@P(``)N"@P"\`%"
MP4+(0LE"TD+30M1"U4+60M="V$+90@X`0@L``"@````@`P``8*7__S0`````
M0@X@1H@$B0:!`D(,"`!8R`P"($+!1,E*#@``1````$P#``!HI?__/@$```!"
M#D!(B`2!`HD&D@A"#`@``N8*#`)`0L%"R$+)0M)"#@!""U`*#`)`0L%"R$+)
M0M)2#@!""P``-````)0#``!>IO__V@,```!"#D!(B`22"),*@0)"#`@``X0#
M"L@,`D!"P4+20M-2#@!$"VR)!@`L````S`,```"J__]B`````$(.,$2(!(D&
M0@P(`$J!`@(^"L%"R`P",$+)0@X`0@L<````_`,``#*J__]>`````$(.8$2(
M%($20@P(0````%`````<!```<*K__UP!````0@YP6H@$@0*)!I((DPJ4#)4.
MEA"7$I@4F1::&$(,"``"V`H,`G!"P4+(0LE"TD+30M1"U4+60M="V$+90MI"
M#@!""R0```!P!```>*O__R@`````0@X01(@$@0)"#`@`6L%"R`P"$$(.``!4
M````F`0``'BK___\`````$(.4$:(!(D&D@A"#`@`8($"9`K(#`)00L%"R4+2
M0@X`1`M0DPH"9,@,`E!"P4+)0M)$TTH.`$0-"($"B`2)!I((0I,*````O```
M`/`$```<K/__/@8```!"#H`#3(@$D@A"#`@`9),*E`R!`HD&`I"5#I80EQ*8
M%)D6`\(!"L%"R`P"@`-"UD+70MA"V4+)0M)"TT+41-5"#@!""P*2FAB;&D;:
M0MMHU=;7V-E6R`P"@`-"P4+20M-"U$3)2@X`0@T(@0*(!(D&D@B3"I0,E0Z6
M$)<2F!29%IH8FQH#9@$*VD+;0@L#'@':VT2:&)L:2M76U]C9VMM.E0Z6$)<2
MF!29%IH8FQH`2````+`%``":L?__)`(```!"#E!"B092B`24#($"D@B5#D(,
M"``#:@$*P4+(#`)00LE"TD+40M5"#@!""UX*DPI4"W@*DPI>"T*3"D0```#\
M!0``<K/__]``````0@Y`1H@$B0:2"$(,"`!,@0("1@K!0L@,`D!"R4+20@X`
M0@L"4@K(#`)`0L%"R4320@X`1`L``#@```!$!@``>(/___0!````0@Y@1H$"
MB`22"$(,"`!$B0:3"@-<`0K!0L@,`F!"R4+21--"#@!""P```"0```"`!@``
MOK/__V``````0@X01(@$@0)"#`@``E+!0L@,`A!"#@!0````J`8``/:S__\:
M`0```$(.8%"(!)((DPJ5#I80@0*)!I0,0@P(``)`F!1(F19&EQ("/M="V$+9
M7`K!0L@,`F!"R4+20M-"U$+50M9"#@!""P!$````_`8``+RT__^^`0```$(.
M4%"(!(D&D@B5#I80@0*3"I0,0@P(``-4`0K(#`)00L%"R4+20M-"U$+50M9"
M#@!""P`````D````1`<``#*V__\:`````$(.$$2(!($"0@P(`$+(#`(00L%*
M#@````````0````0`````0```$=.50``````!`````\`````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````$`````````^0(````````!``````````0#`````````0``````
M```.`P```````!T`````````N`0````````@`````````("K````````(0``
M```````(`````````!D`````````B*L````````;``````````@`````````
M&@````````"0JP```````!P`````````"`````````#U_O]O`````/`"````
M````!0``````````#@````````8`````````B`0````````*`````````-L$
M````````"P`````````8`````````!4````````````````````#````````
M`+BM`````````@`````````P`````````!0`````````!P`````````7````
M`````%@;````````!P`````````(%`````````@`````````@`<````````)
M`````````!@`````````'@`````````(`````````/O__V\``````0``"```
M``#^__]O`````*@3````````____;P`````"`````````/#__V\`````W!(`
M``````#Y__]O``````P`````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````#__________P``````````D!L```````"0&P```````)BK
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````"E!!4BYP;0H`````````````````````````
M````````````````````````````````````````````````````````````
M1T-#.B`H1TY5*2`Q-"XR+C$@,C`R-3`R,#<`1T-#.B`H1TY5*2`Q-2XQ+C$@
M,C`R-3`T,C4`07,```!R:7-C=@`!:0````00!7)V-C1I,G`Q7VTR<#!?83)P
M,5]F,G`R7V0R<#)?8S)P,%]Z:6-S<C)P,%]Z:69E;F-E:3)P,%]Z;6UU;#%P
M,%]Z86%M;S%P,%]Z86QR<V,Q<#!?>F-A,7`P7WIC9#%P,```+G-H<W1R=&%B
M`"YN;W1E+F=N=2YB=6EL9"UI9``N:6YT97)P`"YG;G4N:&%S:``N9'EN<WEM
M`"YD>6YS='(`+F=N=2YV97)S:6]N`"YG;G4N=F5R<VEO;E]R`"YR96QA+F1Y
M;@`N<F5L82YP;'0`+G1E>'0`+G)O9&%T80`N96A?9G)A;65?:&1R`"YE:%]F
M<F%M90`N;F]T92Y!0DDM=&%G`"YP<F5I;FET7V%R<F%Y`"YI;FET7V%R<F%Y
M`"YF:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N
M=``N<FES8W8N871T<FEB=71E<P``````````````````````````````````
M````````````````````````````````````````````````````````````
M``L````'`````@````````"H`@```````*@"````````)```````````````
M``````0````````````````````>`````0````(`````````S`(```````#,
M`@```````"$````````````````````!````````````````````)@```/;_
M_V\"`````````/`"````````\`(```````"4`0````````0`````````"```
M`````````````````#`````+`````@````````"(!````````(@$````````
M>`D````````%`````@````@`````````&``````````X`````P````(`````
M``````X`````````#@```````-L$```````````````````!````````````
M````````0````/___V\"`````````-P2````````W!(```````#*````````
M``0``````````@`````````"`````````$T```#^__]O`@````````"H$P``
M`````*@3````````8``````````%`````@````@```````````````````!<
M````!`````(`````````"!0````````(%````````%`'````````!```````
M```(`````````!@`````````9@````0```!"`````````%@;````````6!L`
M```````P``````````0````4````"``````````8`````````&L````!````
M!@````````"0&P```````)`;````````0````````````````````!``````
M````$`````````!P`````0````8`````````T!L```````#0&P```````-0S
M```````````````````$````````````````````=@````$````"````````
M`*A/````````J$\```````!W00``````````````````"```````````````
M`````'X````!`````@`````````@D0```````""1````````[```````````
M``````````0```````````````````",`````0````(`````````$)(`````
M```0D@```````&P'```````````````````(````````````````````E@``
M``<````"`````````'R9````````?)D````````@````````````````````
M!````````````````````*0````0`````P````````"`JP```````(";````
M````"`````````````````````@`````````"`````````"S````#@````,`
M````````B*L```````"(FP````````@````````````````````(````````
M``@`````````OP````\````#`````````)"K````````D)L````````(````
M````````````````"``````````(`````````,L````&`````P````````"8
MJP```````)B;````````(`(````````%``````````@`````````$```````
M``#4`````0````,`````````N*T```````"XG0```````$@"````````````
M```````(``````````@`````````V0````$````#``````````"P````````
M`*````````!0````````````````````"````````````````````-\````(
M`````P````````!0L````````%"@````````*`````````````````````@`
M``````````````````#D`````0```#````````````````````!0H```````
M`#8````````````````````!``````````$`````````[0````,``'``````
M````````````````AJ````````!T`````````````````````0``````````
M``````````$````#`````````````````````````/J@````````_P``````
:``````````````$`````````````````````
