---
stage: Software Supply Chain Security
group: Pipeline Security
info: To determine the technical writer assigned to the Stage/Group associated with this page, see https://handbook.gitlab.com/handbook/product/ux/technical-writing/#assignments
title: Use Sigstore for keyless signing and verification
---

{{< details >}}

- Tier: Free, Premium, Ultimate
- Offering: GitLab.com

{{< /details >}}

The [Sigstore](https://www.sigstore.dev/) project provides a CLI called
[Cosign](https://docs.sigstore.dev/quickstart/quickstart-cosign/) which can be used for keyless signing of container images built
with GitLab CI/CD. Keyless signing has many advantages, including eliminating the need to manage, safeguard, and rotate a private
key. Cosign requests a short-lived key pair to use for signing, records it on a certificate transparency log, and
then discards it. The key is generated through a token obtained from the GitLab server using the OIDC identity of the user who
ran the pipeline. This token includes unique claims that certify the token was generated by a CI/CD pipeline. To learn more,
see Cosign [documentation](https://docs.sigstore.dev/quickstart/quickstart-cosign/#example-working-with-containers) on keyless signatures.

For details on the mapping between GitLab OIDC claims and Fulcio certificate extensions, see the GitLab column of
[Mapping OIDC token claims to Fulcio OIDs](https://github.com/sigstore/fulcio/blob/main/docs/oid-info.md#mapping-oidc-token-claims-to-fulcio-oids).

Prerequisites:

- You must be using GitLab.com.
- Your project's CI/CD configuration must be located in the project.

## Sign or verify container images and build artifacts by using Cosign

You can use Cosign to sign and verify container images and build artifacts.

Prerequisites:

- You must use a version of Cosign that is `>= 2.0.1`.

**Known issues**

- The `id_tokens` portion of the CI/CD configuration file must be located in the project that is being built and signed. AutoDevOps, CI files included from another repository, and child pipelines are not supported. Work to remove this limitation is being tracked in [epic 11637](https://gitlab.com/groups/gitlab-org/-/epics/11637).

**Best practices**:

- Build and sign an image/artifact in the same job to prevent it from being tampered with before it is signed.
- When signing container images, sign the digest (which is immutable) instead of the tag.

GitLab [ID tokens](../secrets/id_token_authentication.md#id-tokens) can be used by Cosign for
[keyless signing](https://docs.sigstore.dev/quickstart/quickstart-cosign/#keyless-signing-of-a-container). The token must have
`sigstore` set as the [`aud`](../secrets/id_token_authentication.md#token-payload) claim. The token can be used by Cosign automatically when it is set in the
`SIGSTORE_ID_TOKEN` environment variable.

To learn more about how to install Cosign, see [Cosign Installation documentation](https://docs.sigstore.dev/cosign/system_config/installation/).

### Signing

#### Container images

The [`Cosign.gitlab-ci.yml`](https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/templates/Cosign.gitlab-ci.yml)
template can be used to build and sign a container image in GitLab CI. The signature is automatically stored in the same
container repository as the image.

```yaml
include:
- template: Cosign.gitlab-ci.yml
```

To learn more about signing containers, see [Cosign Signing Containers documentation](https://docs.sigstore.dev/cosign/signing/signing_with_containers/).

#### Build artifacts

The following example demonstrates how to sign a build artifact in GitLab CI. You should save the `cosign.bundle` file
produced by `cosign sign-blob`, which is used for signature verification.

To learn more about signing artifacts, see [Cosign Signing Blobs documentation](https://docs.sigstore.dev/cosign/signing/signing_with_blobs/).

```yaml
build_and_sign_artifact:
  stage: build
  image: alpine:latest
  variables:
    COSIGN_YES: "true"
  id_tokens:
    SIGSTORE_ID_TOKEN:
      aud: sigstore
  before_script:
    - apk add --update cosign
  script:
    - echo "This is a build artifact" > artifact.txt
    - cosign sign-blob artifact.txt --bundle cosign.bundle
  artifacts:
    paths:
      - artifact.txt
      - cosign.bundle
```

### Verification

**Command-line arguments**

| Name                        | Value |
|-----------------------------|-------|
| `--certificate-identity`    | The SAN of the signing certificate issued by Fulcio. Can be constructed with the following information from the project where the image/artifact was signed: GitLab instance URL + project path + `//` + CI config path + `@` + ref path. |
| `--certificate-oidc-issuer` | The GitLab instance URL where the image/artifact was signed. For example, `https://gitlab.com`. |
| `--bundle`                  | The `bundle` file produced by `cosign sign-blob`. Only used for verifying build artifacts. |

To learn more about verifying signed images/artifacts, see [Cosign Verifying documentation](https://docs.sigstore.dev/cosign/verifying/verify/).

#### Container images

The following example demonstrates how to verify a signed container image in GitLab CI. Use the previously described [command-line arguments](#verification).

```yaml
verify_image:
  image: alpine:3.20
  stage: verify
  before_script:
    - apk add --update cosign docker
    - docker login -u "$CI_REGISTRY_USER" -p "$CI_REGISTRY_PASSWORD" $CI_REGISTRY
  script:
    - cosign verify "$CI_REGISTRY_IMAGE:$CI_COMMIT_REF_SLUG" --certificate-identity "https://gitlab.com/my-group/my-project//path/to/.gitlab-ci.yml@refs/heads/main" --certificate-oidc-issuer "https://gitlab.com"
```

**Additional details**:

- The double backslash between the project path and the `.gitlab-ci.yml` path is not an error and is required for verification to succeed. A typical error when a single slash is used is `Error: none of the expected identities matched what was in the certificate, got subjects` followed by the signed URL which has two slashes between the project path and the `.gitlab-ci.yml` path.
- If the verification is happening in the same pipeline as the signing, then this path can be used: `"${CI_PROJECT_URL}//.gitlab-ci.yml@refs/heads/${CI_COMMIT_REF_NAME}"`

#### Build artifacts

The following example demonstrates how to verify a signed build artifact in GitLab CI. Verifying an artifact requires both
the artifact itself and the `cosign.bundle` file produced by `cosign sign-blob`. Use the previously described [command-line arguments](#verification).

```yaml
verify_artifact:
  stage: verify
  image: alpine:latest
  before_script:
    - apk add --update cosign
  script:
    - cosign verify-blob artifact.txt --bundle cosign.bundle --certificate-identity "https://gitlab.com/my-group/my-project//path/to/.gitlab-ci.yml@refs/heads/main" --certificate-oidc-issuer "https://gitlab.com"
```

**Additional details**:

- The double backslash between the project path and the `.gitlab-ci.yml` path is not an error and is required for verification to succeed. A typical error when a single slash is used is `Error: none of the expected identities matched what was in the certificate, got subjects` followed by the signed URL which has two slashes between the project path and the `.gitlab-ci.yml` path.
- If the verification is happening in the same pipeline as the signing, then this path can be used: `"${CI_PROJECT_URL}//.gitlab-ci.yml@refs/heads/${CI_COMMIT_REF_NAME}"`

## Use Sigstore and npm to generate keyless provenance

You can use Sigstore and npm, together with GitLab CI/CD, to digitally sign build artifacts without the overhead of key management.

### About npm provenance

[npm CLI](https://docs.npmjs.com/cli/) allows package maintainers to provide users with provenance attestations. Using npm
CLI provenance generation allows users to trust and verify that the package they are downloading and using is from you and the
build system that built it.

For more information on how to publish npm packages, see [GitLab npm package registry](../../user/packages/npm_registry/_index.md).

### Sigstore

[Sigstore](https://www.sigstore.dev/) is a set of tools that package managers and security experts can use to secure their software
supply chains against attacks. Bringing together free-to-use open source technologies like Fulcio, Cosign, and Rekor, it
handles digital signing, verification, and checks for provenance
needed to make it safer to distribute and use open source software.

**Related topics**:

- [SLSA Provenance definition](https://slsa.dev/provenance/v1)
- [npm documentation](https://docs.npmjs.com/generating-provenance-statements/)
- [npm Provenance RFC](https://github.com/npm/rfcs/blob/main/accepted/0049-link-packages-to-source-and-build.md#detailed-steps-to-publish)

### Generating provenance in GitLab CI/CD

Now that Sigstore supports GitLab OIDC as previously described, you can use npm provenance together with GitLab CI/CD and Sigstore to
generate and sign provenance for your npm packages in a GitLab CI/CD pipeline.

#### Prerequisites

1. Set your GitLab [ID token](../secrets/id_token_authentication.md) `aud` to `sigstore`.
1. Add the `--provenance` flag to have npm publish.

Example content to be added to `.gitlab-ci.yml` file:

```yaml
build:
  image: node:latest
  id_tokens:
    SIGSTORE_ID_TOKEN:
      aud: sigstore
  script:
    - npm publish --provenance --access public
```

The npm GitLab template provides this functionality as well, the example is in
the [templates documentation](https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/templates/npm.gitlab-ci.yml).

## Verifying npm provenance

npm CLI also provides functionality for end users to verify the provenance of packages.

```plaintext
npm audit signatures
audited 1 package in 0s
1 package has a verified registry signature
```

### Inspecting the provenance metadata

The Rekor transparency log stores certificates and attestations for every package that is published with provenance.
For example, here is the [entry for the following example](https://search.sigstore.dev/?logIndex=21076013).

An example provenance document generated by npm:

```yaml
_type: https://in-toto.io/Statement/v0.1
subject:
  - name: pkg:npm/%40strongjz/strongcoin@0.0.13
    digest:
      sha512: >-
        924a134a0fd4fe6a7c87b4687bf0ac898b9153218ce9ad75798cc27ab2cddbeff77541f3847049bd5e3dfd74cea0a83754e7686852f34b185c3621d3932bc3c8
predicateType: https://slsa.dev/provenance/v0.2
predicate:
  buildType: https://github.com/npm/CLI/gitlab/v0alpha1
  builder:
    id: https://gitlab.com/strongjz/npm-provenance-example/-/runners/12270835
  invocation:
    configSource:
      uri: git+https://gitlab.com/strongjz/npm-provenance-example
      digest:
        sha1: 6e02e901e936bfac3d4691984dff8c505410cbc3
      entryPoint: deploy
    parameters:
      CI: 'true'
      CI_API_GRAPHQL_URL: https://gitlab.com/api/graphql
      CI_API_V4_URL: https://gitlab.com/api/v4
      CI_COMMIT_BEFORE_SHA: 7d3e913e5375f68700e0c34aa90b0be7843edf6c
      CI_COMMIT_BRANCH: main
      CI_COMMIT_REF_NAME: main
      CI_COMMIT_REF_PROTECTED: 'true'
      CI_COMMIT_REF_SLUG: main
      CI_COMMIT_SHA: 6e02e901e936bfac3d4691984dff8c505410cbc3
      CI_COMMIT_SHORT_SHA: 6e02e901
      CI_COMMIT_TIMESTAMP: '2023-05-19T10:17:12-04:00'
      CI_COMMIT_TITLE: trying to publish to gitlab reg
      CI_CONFIG_PATH: .gitlab-ci.yml
      CI_DEFAULT_BRANCH: main
      CI_DEPENDENCY_PROXY_DIRECT_GROUP_IMAGE_PREFIX: gitlab.com:443/strongjz/dependency_proxy/containers
      CI_DEPENDENCY_PROXY_GROUP_IMAGE_PREFIX: gitlab.com:443/strongjz/dependency_proxy/containers
      CI_DEPENDENCY_PROXY_SERVER: gitlab.com:443
      CI_DEPENDENCY_PROXY_USER: gitlab-ci-token
      CI_JOB_ID: '4316132595'
      CI_JOB_NAME: deploy
      CI_JOB_NAME_SLUG: deploy
      CI_JOB_STAGE: deploy
      CI_JOB_STARTED_AT: '2023-05-19T14:17:23Z'
      CI_JOB_URL: https://gitlab.com/strongjz/npm-provenance-example/-/jobs/4316132595
      CI_NODE_TOTAL: '1'
      CI_PAGES_DOMAIN: gitlab.io
      CI_PAGES_URL: https://strongjz.gitlab.io/npm-provenance-example
      CI_PIPELINE_CREATED_AT: '2023-05-19T14:17:21Z'
      CI_PIPELINE_ID: '872773336'
      CI_PIPELINE_IID: '40'
      CI_PIPELINE_SOURCE: push
      CI_PIPELINE_URL: https://gitlab.com/strongjz/npm-provenance-example/-/pipelines/872773336
      CI_PROJECT_CLASSIFICATION_LABEL: ''
      CI_PROJECT_DESCRIPTION: ''
      CI_PROJECT_ID: '45821955'
      CI_PROJECT_NAME: npm-provenance-example
      CI_PROJECT_NAMESPACE: strongjz
      CI_PROJECT_NAMESPACE_SLUG: strongjz
      CI_PROJECT_NAMESPACE_ID: '36018'
      CI_PROJECT_PATH: strongjz/npm-provenance-example
      CI_PROJECT_PATH_SLUG: strongjz-npm-provenance-example
      CI_PROJECT_REPOSITORY_LANGUAGES: javascript,dockerfile
      CI_PROJECT_ROOT_NAMESPACE: strongjz
      CI_PROJECT_TITLE: npm-provenance-example
      CI_PROJECT_URL: https://gitlab.com/strongjz/npm-provenance-example
      CI_PROJECT_VISIBILITY: public
      CI_REGISTRY: registry.gitlab.com
      CI_REGISTRY_IMAGE: registry.gitlab.com/strongjz/npm-provenance-example
      CI_REGISTRY_USER: gitlab-ci-token
      CI_RUNNER_DESCRIPTION: 3-blue.shared.runners-manager.gitlab.com/default
      CI_RUNNER_ID: '12270835'
      CI_RUNNER_TAGS: >-
        ["gce", "east-c", "linux", "ruby", "mysql", "postgres", "mongo",
        "git-annex", "shared", "docker", "saas-linux-small-amd64"]
      CI_SERVER_HOST: gitlab.com
      CI_SERVER_NAME: GitLab
      CI_SERVER_PORT: '443'
      CI_SERVER_PROTOCOL: https
      CI_SERVER_REVISION: 9d4873fd3c5
      CI_SERVER_SHELL_SSH_HOST: gitlab.com
      CI_SERVER_SHELL_SSH_PORT: '22'
      CI_SERVER_URL: https://gitlab.com
      CI_SERVER_VERSION: 16.1.0-pre
      CI_SERVER_VERSION_MAJOR: '16'
      CI_SERVER_VERSION_MINOR: '1'
      CI_SERVER_VERSION_PATCH: '0'
      CI_TEMPLATE_REGISTRY_HOST: registry.gitlab.com
      GITLAB_CI: 'true'
      GITLAB_FEATURES: >-
        elastic_search,ldap_group_sync,multiple_ldap_servers,seat_link,usage_quotas,zoekt_code_search,repository_size_limit,admin_audit_log,auditor_user,custom_file_templates,custom_project_templates,db_load_balancing,default_branch_protection_restriction_in_groups,extended_audit_events,external_authorization_service_api_management,geo,instance_level_scim,ldap_group_sync_filter,object_storage,pages_size_limit,project_aliases,password_complexity,enterprise_templates,git_abuse_rate_limit,required_ci_templates,runner_maintenance_note,runner_performance_insights,runner_upgrade_management,runner_jobs_statistics
      GITLAB_USER_ID: '31705'
      GITLAB_USER_LOGIN: strongjz
    environment:
      name: 3-blue.shared.runners-manager.gitlab.com/default
      architecture: linux/amd64
      server: https://gitlab.com
      project: strongjz/npm-provenance-example
      job:
        id: '4316132595'
      pipeline:
        id: '872773336'
        ref: .gitlab-ci.yml
  metadata:
    buildInvocationId: https://gitlab.com/strongjz/npm-provenance-example/-/jobs/4316132595
    completeness:
      parameters: true
      environment: true
      materials: false
    reproducible: false
  materials:
    - uri: git+https://gitlab.com/strongjz/npm-provenance-example
      digest:
        sha1: 6e02e901e936bfac3d4691984dff8c505410cbc3
```
