"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineResourceAccessManagementRoutes = defineResourceAccessManagementRoutes;
var _configSchema = require("@osd/config-schema");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

/* ******************* Schemas ******************* */

const stringArray = _configSchema.schema.arrayOf(_configSchema.schema.string());

// Inner object allowed on each access level
const recipientsSchema = _configSchema.schema.object({
  users: _configSchema.schema.maybe(stringArray),
  roles: _configSchema.schema.maybe(stringArray),
  backend_roles: _configSchema.schema.maybe(stringArray)
}, {
  unknowns: 'forbid',
  validate: v => {
    if (!v.users && !v.roles && !v.backend_roles) {
      return 'Each share_with entry must include at least one of "users", "roles", or "backend_roles".';
    }
  }
});

// share_with shape used by both PUT and PATCH (add/revoke)
const shareWithSchema = _configSchema.schema.recordOf(_configSchema.schema.string(), recipientsSchema);

// PUT body must include non-empty share_with
const putBodySchema = _configSchema.schema.object({
  resource_id: _configSchema.schema.string({
    minLength: 1
  }),
  resource_type: _configSchema.schema.string({
    minLength: 1
  }),
  share_with: shareWithSchema
}, {
  unknowns: 'allow',
  validate: val => {
    if (Object.keys(val.share_with).length === 0) {
      return '"share_with" must not be empty.';
    }
  }
});

// PATCH schema: add/revoke must be shareWithSchema
const patchBodySchema = _configSchema.schema.object({
  resource_id: _configSchema.schema.string({
    minLength: 1
  }),
  resource_type: _configSchema.schema.string({
    minLength: 1
  }),
  add: _configSchema.schema.maybe(shareWithSchema),
  revoke: _configSchema.schema.maybe(shareWithSchema)
}, {
  validate: value => {
    if (!value.add && !value.revoke) {
      return 'Request body must include at least one of "add" or "revoke".';
    }
  }
});

/* *******************Route definitions******************* */

const SECURITY_RESOURCE_API_PREFIX = '/_plugins/_security/api/resource';
const LIST_TYPES_API = `${SECURITY_RESOURCE_API_PREFIX}/types`;
const LIST_SHARING_INFO_API = `${SECURITY_RESOURCE_API_PREFIX}/list`;
const SHARE_API = `${SECURITY_RESOURCE_API_PREFIX}/share`;
function defineResourceAccessManagementRoutes(router) {
  // GET registered resource types
  router.get({
    path: '/api/resource/types',
    validate: false
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'GET',
        path: LIST_TYPES_API
      });
      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // GET accessible shared resources filtered by resourceType
  router.get({
    path: '/api/resource/list',
    validate: {
      query: _configSchema.schema.object({
        resourceType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const {
        resourceType
      } = request.query;
      const result = await client.transport.request({
        method: 'GET',
        path: LIST_SHARING_INFO_API,
        querystring: {
          resource_type: resourceType
        } // backend expects snake_case
      });

      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // GET sharing info for a specific resource
  router.get({
    path: '/api/resource/view',
    validate: {
      query: _configSchema.schema.object({
        resourceId: _configSchema.schema.string(),
        resourceType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const {
        resourceId,
        resourceType
      } = request.query;
      const result = await client.transport.request({
        method: 'GET',
        path: SHARE_API,
        querystring: {
          resource_id: resourceId,
          resource_type: resourceType
        } // backend expects snake_case
      });

      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // PUT share a resource — requires `share_with`
  router.put({
    path: '/api/resource/share',
    validate: {
      body: putBodySchema
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'PUT',
        path: SHARE_API,
        body: request.body
      });
      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // PATCH update sharing — `add`/`revoke` adhere to share_with schema
  router.patch({
    path: '/api/resource/update_sharing',
    validate: {
      body: patchBodySchema
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'PATCH',
        path: SHARE_API,
        body: request.body
      });
      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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