#! /usr/bin/env python3
# -*- coding: utf-8 -*-

# Copyright (c) 2011-2012, The Linux Foundation. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the name of The Linux Foundation nor
#       the names of its contributors may be used to endorse or promote
#       products derived from this software without specific prior written
#       permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NON-INFRINGEMENT ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
# OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# Invoke gcc, looking for warnings, and causing a failure if there are
# non-whitelisted warnings.

from __future__ import print_function
import errno
import re
import os
import sys
import subprocess

allowed_warnings = set([
    "cgroup.h:35", # include/trace/hooks/cgroup.h:35:18: warning: 'struct cgroup_subsys' declared inside parameter list will not be visible outside of this definition or declaration
    "vfs.c:1259", # fs/incfs/vfs.c:1259:10: warning: cast to pointer from integer of different size [-Wint-to-pointer-cast]
    "pseudo_files.c:655", # fs/incfs/pseudo_files.c:655:10: warning: cast to pointer from integer of different size [-Wint-to-pointer-cast]
    "km_apphint.c:230", # drivers/staging/imgtec/rogue/km_apphint.c:230:48: warning: division 'sizeof (void *) / sizeof (void)' does not compute the number of array elements [-Wsizeof-pointer-div]
    "file.c:3010", # fs/f2fs/file.c:3010:12: warning: 'f2fs_ioctl_check_project' defined but not used
    "configfs.c:1488", # drivers/usb/gadget/configfs.c:1488:12: warning: 'configfs_composite_setup' defined but not used
    "configfs.c:1513", # drivers/usb/gadget/configfs.c:1513:13: warning: 'configfs_composite_disconnect' defined but not used
    "posix-cpu-timers.c:1268", # kernel/time/posix-cpu-timers.c:1268:13: warning: 'now' may be used uninitialized in this function
    "af_unix.c:1036", # net/unix/af_unix.c:1036:20: warning: 'hash' may be used uninitialized in this function
    "sunxi_sram.c:214", # drivers/soc/sunxi/sunxi_sram.c:214:24: warning: 'device' may be used uninitialized in this function
    "ks8851.c:298", # drivers/net/ethernet/micrel/ks8851.c:298:2: warning: 'rxb[0]' may be used uninitialized in this function
    "ks8851.c:421", # drivers/net/ethernet/micrel/ks8851.c:421:20: warning: 'rxb[0]' may be used uninitialized in this function
    "compat_binfmt_elf.c:58", # fs/compat_binfmt_elf.c:58:13: warning: 'cputime_to_compat_timeval' defined but not used
    "memcontrol.c:5337", # mm/memcontrol.c:5337:12: warning: initialization from incompatible pointer type
    "atags_to_fdt.c:101", # arch/arm/boot/compressed/atags_to_fdt.c:101:1: warning: the frame size of 2056 bytes is larger than 1280 bytes [-Wframe-larger-than=]
    "drm_edid.c:3506", # drivers/gpu/drm/drm_edid.c:3506:13: warning: 'cea_db_is_hdmi_forum_vsdb' defined but not used
    # W=1
    "bounds.c:15", # kernel/bounds.c:15:6: warning: no previous prototype for ‘foo’
    "cpufeature.h:157", # arch/arm64/include/asm/cpufeature.h:157:68: warning: signed and unsigned type in conditional expression
    "sched.h:1211", # include/linux/sched.h:1211:1: warning: type qualifiers ignored on function return type
    "halphyrf_8188e_ce.c:2208", # drivers/net/wireless/rockchip_wlan/rtl8189es/hal/phydm/rtl8188e/halphyrf_8188e_ce.c:2208:1: warning: the frame size of 1056 bytes is larger than 1024 bytes
    "halphyrf_8723b_ce.c:2879", # drivers/net/wireless/rockchip_wlan/rtl8723bu/hal/phydm/rtl8723b/halphyrf_8723b_ce.c:2879:1: warning: the frame size of 1056 bytes is larger than 1024 bytes
 ])

# Capture the name of the object file, can find it.
ofile = None

do_exit = False;

warning_re = re.compile(r'''(.*/|)([^/]+\.[a-z]+:\d+):(\d+:)? warning:''')
def interpret_warning(line):
    """Decode the message from gcc.  The messages we care about have a filename, and a warning"""
    line = line.rstrip('\n')
    m = warning_re.match(line)
    if m and m.group(2) not in allowed_warnings:
        print ("error, forbidden warning:" + m.group(2))

        # If there is a warning, remove any object if it exists.
        if ofile:
            try:
                os.remove(ofile)
            except OSError:
                pass
        global do_exit
        do_exit = True;

def run_gcc():
    args = sys.argv[1:]
    # Look for -o
    try:
        i = args.index('-o')
        global ofile
        ofile = args[i+1]
    except (ValueError, IndexError):
        pass

    compiler = sys.argv[0]

    try:
        env = os.environ.copy()
        env['LC_ALL'] = 'C'
        proc = subprocess.Popen(args, stderr=subprocess.PIPE, env=env)
        for line in proc.stderr:
            print (line.decode("utf-8"), end="")
            interpret_warning(line.decode("utf-8"))
        if do_exit:
            sys.exit(1)

        result = proc.wait()
    except OSError as e:
        result = e.errno
        if result == errno.ENOENT:
            print (args[0] + ':' + e.strerror)
            print ('Is your PATH set correctly?')
        else:
            print (' '.join(args) + str(e))

    return result

if __name__ == '__main__':
    status = run_gcc()
    sys.exit(status)
